package com.bxm.pangu.rta.common.zhihu;

import com.bxm.pangu.rta.common.*;
import com.google.protobuf.InvalidProtocolBufferException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ByteArrayEntity;

import java.util.List;

/**
 * @author allen
 * @date 2022-01-11
 * @since 1.0
 */
@Slf4j
public class ZhihuRtaClient extends AbstractHttpClientRtaClient {

    private final ZhihuRtaProperties properties;

    public ZhihuRtaClient(ZhihuRtaProperties properties) {
        super(properties);
        this.properties = properties;
    }

    @Override
    protected HttpRequestBase create(RtaRequest request) {
        // {slot_id}|{rta_ids}|{identify}|{url}
        String param = request.getParam();
        if (StringUtils.isBlank(param)) {
            throw new RtaRequestException("param must be {slot_id|rta_ids}");
        }
        String[] arr = StringUtils.splitByWholeSeparator(param, "|");
        if (arr.length < 2) {
            throw new RtaRequestException("param must be {slot_id|rta_ids}");
        }
        String slotId = arr[0];
        String rtaIds = arr[1];

        String identify = properties.getIdentify();
        if (arr.length >= 3) {
            identify = arr[2];
        }
        String url = getProperties().getUrl();
        if (arr.length >= 4) {
            url = arr[3];
        }

        HttpPost post = new HttpPost(url);

        Zhihu.Req.Builder builder = Zhihu.Req.newBuilder();

        String imei = request.getImei();
        String imeiMd5 = request.getImei_md5();
        String androidid = request.getAndroidid();
        String androididMd5 = request.getAndroidid_md5();
        String oaidMd5 = request.getOaid_md5();
        String oaid = request.getOaid();
        String idfa = request.getIdfa();
        String idfaMd5 = request.getIdfa_md5();
        if (StringUtils.isNotBlank(imei)) {
            builder.setDid(imei)
                    .setDidType(Zhihu.DidType.IMEI);
        } else if (StringUtils.isNotBlank(imeiMd5)) {
            builder.setDid(imeiMd5)
                    .setDidType(Zhihu.DidType.IMEI_MD5);
        } else if (StringUtils.isNotBlank(androidid)) {
            builder.setDid(androidid)
                    .setDidType(Zhihu.DidType.ANDROID_ID);
        } else if (StringUtils.isNotBlank(androididMd5)) {
            builder.setDid(androididMd5)
                    .setDidType(Zhihu.DidType.ANDROID_ID_MD5);
        } else if (StringUtils.isNotBlank(oaid)) {
            builder.setDid(oaid)
                    .setDidType(Zhihu.DidType.OAID);
        } else if (StringUtils.isNotBlank(oaidMd5)) {
            builder.setDid(oaidMd5)
                    .setDidType(Zhihu.DidType.OAID_MD5);
        } else if (StringUtils.isNotBlank(idfa)) {
            builder.setDid(idfa)
                    .setDidType(Zhihu.DidType.IDFA);
        } else if (StringUtils.isNotBlank(idfaMd5)) {
            builder.setDid(idfaMd5)
                    .setDidType(Zhihu.DidType.IDFA_MD5);
        }

        Zhihu.Device.Builder deviceBuilder = Zhihu.Device.newBuilder();
        if (StringUtils.isNotBlank(imei)) {
            deviceBuilder.setImeiMd5(DigestUtils.md5Hex(imei).toLowerCase());
        }
        if (StringUtils.isNotBlank(imeiMd5)) {
            deviceBuilder.setImeiMd5(imeiMd5);
        }
        if (StringUtils.isNotBlank(androididMd5)) {
            deviceBuilder.setAndroidIdMd5(androididMd5);
        }
        if (StringUtils.isNotBlank(oaid)) {
            deviceBuilder.setOaid(oaid);
        }
        if (StringUtils.isNotBlank(oaidMd5)) {
            deviceBuilder.setOaidMd5(oaidMd5);
        }
        if (StringUtils.isNotBlank(idfa)) {
            deviceBuilder.setIdfa(idfa);
        }
        if (StringUtils.isNotBlank(idfaMd5)) {
            deviceBuilder.setIdfaMd5(idfaMd5);
        }

        builder.setDevice(deviceBuilder.build());



        builder.setSlotId(NumberUtils.toLong(slotId));

        String reqId = RandomStringUtils.randomAlphanumeric(8);
        builder.setReqId(reqId);

        builder.addRtaIds(NumberUtils.toLong(rtaIds));

        long currentTimeMillis = System.currentTimeMillis();
        builder.setSignTime(currentTimeMillis);



        String token = DigestUtils.md5Hex(reqId + currentTimeMillis + identify);
        builder.setToken(token);

        post.addHeader("Content-Type", "application/x-protobuf");
        Zhihu.Req build = builder.build();
        post.setEntity(new ByteArrayEntity(build.toByteArray()));

        return post;
    }

    @Override
    protected boolean isTarget(RtaRequest request, String json) {
        throw new UnsupportedOperationException();
    }

    @Override
    public RtaType getRtaType() {
        return RtaType.Zhihu;
    }

    @Override
    protected boolean isTarget(RtaRequest request, byte[] bytes) {
        try {
            Zhihu.Rsp rsp = Zhihu.Rsp.parseFrom(bytes);
            if (rsp.getStatusCode() != 0) {
                return false;
            }
            if (rsp.getBidType() != 0) {
                return false;
            }
            List<Zhihu.UserInfo> userInfosList = rsp.getUserInfosList();
            if (CollectionUtils.isEmpty(userInfosList)) {
                return false;
            }
            Zhihu.UserInfo userInfo = userInfosList.get(0);
            return userInfo.isInitialized();
        } catch (InvalidProtocolBufferException e) {
            log.error("isTarget: ", e);
            return false;
        }
    }

    @Override
    public ProtocolFormat getProtocolFormat() {
        return ProtocolFormat.Protobuf;
    }
}
