package com.bxm.pangu.rta.common.bzy;

import com.bxm.pangu.rta.common.*;
import com.bxm.pangu.rta.common.utils.HttpClientUtils;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UUIDHelper;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;
import org.springframework.http.MediaType;

import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;

/**
 * 暴走鱼rta
 * @author jingyu.li
 * @date 2024-04-23
 */
@Slf4j
public class BzyRtaClient implements RtaClient {

    private final BzyRtaProperties properties;
    private final HttpClient httpClient;

    public BzyRtaClient(BzyRtaProperties properties) {
        this.properties = properties;
        this.httpClient = HttpClientUtils.createHttpClient(properties.getMaxTotal(), properties.getDefaultMaxPerRoute(),
                properties.getConnectionRequestTimeout(), properties.getConnectTimeout(), properties.getSocketTimeout());
    }

    @Override
    public boolean isTarget(RtaRequest request, Consumer<Map<Object, Object>> feedback) throws RtaRequestException {
        HttpRequestBase requestBase = null;
        RtaType rtaType = getRtaType();
        try {
            requestBase = create(request);
            HttpResponse response = httpClient.execute(requestBase);
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode != HttpStatus.SC_OK) {
                log.warn("{} - statusCode: {}", rtaType, statusCode);
            }

            String body = EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8);
            return isTarget(body,feedback);
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("{} - {}", rtaType, e.getMessage());
            }
            throw new RtaRequestException(e);
        } finally {
            if (Objects.nonNull(requestBase) && !requestBase.isAborted()) {
                requestBase.abort();
            }
        }
    }

    private HttpRequestBase create(RtaRequest request) {
        String param = request.getParam();
        if (StringUtils.isBlank(param)) {
            log.warn("bzyRtaWarn param must contain sspid|task_id|channel_id");
            throw new RtaRequestException("param must contain sspid|task_id");
        }
        String[] params = param.split("\\|");
        if (params.length < 2) {
            log.warn("bzyRtaWarn param must contain sspid|task_id|channel_id");
            throw new RtaRequestException("param must contain sspid|task_id");
        }

        BzyRequest.DeviceInfo device = new BzyRequest.DeviceInfo();

        String ua = request.getExt();
        if (StringUtils.isBlank(ua)) {
            ua = "ua_miss";
        }
        device.setUa(ua);

        String imei = request.getImei();
        String oaid = request.getOaid();
        String idfa = request.getIdfa();
        String imei_md5 = request.getImei_md5();
        String oaid_md5 = request.getOaid_md5();
        String idfa_md5 = request.getIdfa_md5();
        if (StringUtils.isNotBlank(imei)) {
            device.setImei(imei);
        } else if (StringUtils.isNotBlank(oaid)) {
            device.setOaid(oaid);
        } else if (StringUtils.isNotBlank(idfa)) {
            device.setIdfa(idfa);
        } else if (StringUtils.isNotBlank(imei_md5)) {
            device.setImei_md5(imei_md5);
        } else if (StringUtils.isNotBlank(oaid_md5)) {
            device.setOaid_md5(oaid_md5);
        } else if (StringUtils.isNotBlank(idfa_md5)) {
            device.setIdfa_md5(idfa_md5);
        } else {
            log.warn("bzyRtaWarn deviceId is empty");
            throw new RtaRequestException("deviceId");
        }

        String sspid = params[0];
        String task_id = params[1];

        BzyRequest.Adslot adslot = new BzyRequest.Adslot();
        adslot.setTask_id(task_id);
        if (params.length > 2) {
            adslot.setChannel_id(params[2]);
        }

        BzyRequest bzyRequest = new BzyRequest();
        String requestId = UUIDHelper.generate();
        bzyRequest.setRequest_id(requestId);
        bzyRequest.setSspid(sspid);
        bzyRequest.setDevice(device);
        bzyRequest.setAdslot(adslot);
        bzyRequest.setSign(sign(sspid,requestId));

        HttpPost post = new HttpPost(properties.getUrl());
        post.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE);
        post.setEntity(new StringEntity(JsonHelper.convert(bzyRequest),StandardCharsets.UTF_8));
        return post;
    }

    /**
     * 签名
     */
    private String sign(String sspid, String requestId) {
        return DigestUtils.md5Hex(sspid + properties.getToken() + requestId);
    }

    /**
     * 判断返回值
     * @param json  返回数据
     * @param feedback  对返回值处理逻辑
     */
    private boolean isTarget(String json,Consumer<Map<Object, Object>> feedback) {
        if (StringUtils.isBlank(json)) {
            return false;
        }
        BzyResponse bzyResponse = JsonHelper.convert(json, BzyResponse.class);
        if (bzyResponse.getError_code() != 0) {
            return false;
        }
        BzyResponse.Ads ads = bzyResponse.getAds();
        if (ads != null && feedback != null) {
            Map<Object, Object> res = Maps.newHashMap();
            res.put(Constants.DEEP_LINK,ads.getDeep_link());
            res.put(Constants.H5_URL,ads.getH5_url());
            res.put(Constants.AD_IMPRESSION,ads.getAd_impression());
            res.put(Constants.AD_CLICK,ads.getAd_click());
            feedback.accept(res);
        }
        return true;
    }

    @Override
    public RtaType getRtaType() {
        return RtaType.Bzy;
    }

    @Override
    public RtaClientProperties getProperties() {
        return properties;
    }
}
