package com.bxm.pangu.rta.common.alipay;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bxm.pangu.rta.common.*;
import com.bxm.warcar.utils.UUIDHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.List;
import java.util.Objects;

/**
 * @author huxiao
 * @date 2024/7/10
 * @since 1.0.0
 */
@Slf4j
public class AlipayRtaClient extends AbstractHttpClientRtaClient {

    public AlipayRtaClient(AbstractRtaProperties properties) {
        super(properties);
    }

    @Override
    protected HttpRequestBase create(RtaRequest request) {
        // https://ugapi.alipay.com/rta/json/{mediaName}

        // {
        //    "devices": {
        //        "e8bb35755e3c3a88ebd77bb0a22f813a": {
        //            "OAID": {
        //                "encrypt_type": "MD5",
        //                "device_id": "8cf4ebb0472be5bb035b2dbf863a9196"
        //            }
        //        }
        //    },
        //    "request_id": "6f67871d40652a4edb1c92ee1a40b280",
        //    "rta_id_list": ["PID2088531974870077"]
        // }
        String param = request.getParam();
        if (StringUtils.isBlank(param)) {
            throw new RtaRequestException("Illegal param value");
        }
        String[] strings = StringUtils.split(param, "|");

        String mediaName = strings[0];

        String url = getProperties().getUrl();
        url = UriComponentsBuilder.fromUriString(url).replacePath("/rta/json/" + mediaName).build().toUriString();

        JSONObject req = new JSONObject();
        req.put("request_id", UUIDHelper.generate());

        if (strings.length >= 2) {
            String ids = strings[1];
            req.put("rta_id_list", ids.split(","));
        }

        String uniqueKey = UUIDHelper.generate();

        String deviceType = "";
        String deviceId = "";
        String encryptType = "";

        if (StringUtils.isNotBlank(request.getOaid_md5())) {
            deviceType = "OAID";
            encryptType = "MD5";
            deviceId = request.getOaid_md5();
        } else if (StringUtils.isNotBlank(request.getOaid())) {
            deviceType = "OAID";
            deviceId = request.getOaid();
        } else if (StringUtils.isNotBlank(request.getImei_md5())) {
            deviceType = "IMEI";
            encryptType = "MD5";
            deviceId = request.getImei_md5();
        } else if (StringUtils.isNotBlank(request.getImei())) {
            deviceType = "IMEI";
            deviceId = request.getImei();
        } else if (StringUtils.isNotBlank(request.getIdfa_md5())) {
            deviceType = "IDFA";
            encryptType = "MD5";
            deviceId = request.getIdfa_md5();
        } else if (StringUtils.isNotBlank(request.getIdfa())) {
            deviceType = "IDFA";
            deviceId = request.getIdfa();
        } else if (CollectionUtils.isNotEmpty(request.getCaid_info())) {
            // 取最大的版本
            List<RtaRequest.Caid> caidInfo = request.getCaid_info();
            caidInfo.sort((o1, o2) -> {
                int v1 = NumberUtils.toInt(o1.getCaid_version());
                int v2 = NumberUtils.toInt(o2.getCaid_version());
                return Integer.compare(v2, v1);
            });
            RtaRequest.Caid caid = caidInfo.get(0);
            deviceType = "CAID";
            deviceId = caid.getCaid() + "_" + caid.getCaid_version();
            if (caid.isMd5()) {
                encryptType = "MD5";
            }
        }
        JSONObject info = new JSONObject();
        info.put("encrypt_type", encryptType);
        info.put("device_id", deviceId);

        JSONObject device = new JSONObject();
        device.put(deviceType, info);

        JSONObject devices = new JSONObject();
        devices.put(uniqueKey, device);

        req.put("devices", devices);

        HttpPost post = new HttpPost(url);
        post.setEntity(new StringEntity(req.toJSONString(), "UTF-8"));
        post.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE);
        return post;
    }

    /**
     * {
     *     "extMap": {},
     *     "retriable": false,
     *     "response": {
     *         "request": {
     *             "devices": {
     *                 "1696758809001": {
     *                     "OAID": {
     *                         "device_id": "0ff44e276ede86eb"
     *                     }
     *                 }
     *             }
     *         },
     *         "principalLabel": "L00002",
     *         "rtaInfoList": [
     *             {
     *                 "accountId": "L00002"
     *             }
     *         ],
     *         "requiredFlow": true,
     *         "biddingScoreMap": {}
     *     },
     *     "success": true,
     *     "resultCode": "SUCCESS",
     *     "resultDesc": "成功"
     * }
     * @param request 请求对象
     * @param json 返回内容
     * @return
     */
    @Override
    protected boolean isTarget(RtaRequest request, String json) {
        // {
        //    "extMap": {},
        //    "retriable": false,
        //    "response": {
        //        "requiredFlow": false,
        //        "biddingScoreMap": {}
        //    },
        //    "success": true,
        //    "resultCode": "SUCCESS",
        //    "resultDesc": "成功"
        // }
        String accountId = getAccountId(request);
        if (StringUtils.isEmpty(accountId)) {
            log.warn("accountId empty, param = {}", request.getParam());
            return false;
        }
        JSONObject jsonObject = JSONObject.parseObject(json);
        Boolean success = jsonObject.getBoolean("success");
        if (success != null && success) {
            JSONObject response = jsonObject.getJSONObject("response");
            if (response != null) {
                boolean required = response.getBoolean("requiredFlow");
                if (required) {
                    if (response.containsKey("rtaInfoList")) {
                        JSONArray rtaInfoList = response.getJSONArray("rtaInfoList");
                        for (int i = 0; i < rtaInfoList.size(); i++) {
                            JSONObject account = rtaInfoList.getJSONObject(i);
                            if (Objects.nonNull(account)) {
                                String id = account.getString("accountId");
                                boolean result = StringUtils.isNotBlank(id) && accountId.equals(id);
                                if (result) {
                                    return result;
                                }
                            }
                        }
                    } else {
                        log.warn("rtaInfoList not exist, response = {}", json);
                    }
                }
            }
        }
        return false;
    }

    /**
     * 规则：{mediaName}|{rta_id_list,rta_id_list}|{rtaInfo}
     * 示例：bianxianmao|PID2088531974870077,PID2088531974870078|L00002
     * 说明：rta_id_list 非必填。rtaInfo 非必填，如果填了将与响应的 rtaInfoList 的 accountId 匹配，符合其中之一才表示命中。
     *
     * @param request
     * @return
     */
    private String getAccountId(RtaRequest request) {
        String param = request.getParam();
        if (StringUtils.isEmpty(param)) {
            return StringUtils.EMPTY;
        }
        String[] array = param.split("\\|");
        if (array.length < 3) {
            return StringUtils.EMPTY;
        }
        return array[2];
    }

    @Override
    public RtaType getRtaType() {
        return RtaType.Alipay;
    }
}
