package com.bxm.pangu.rta.common.getui;

import com.bxm.pangu.rta.common.AbstractHttpClientRtaClient;
import com.bxm.pangu.rta.common.RtaRequest;
import com.bxm.pangu.rta.common.RtaRequestException;
import com.bxm.pangu.rta.common.RtaType;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.TypeHelper;
import com.bxm.warcar.utils.UUIDHelper;
import com.google.common.collect.Lists;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.springframework.http.MediaType;

import java.nio.charset.StandardCharsets;
import java.util.List;

/**
 * @author allen
 * @date 2021-12-20
 * @since 1.0
 */
public class GetuiRtaClient extends AbstractHttpClientRtaClient {

    private final GetuiRtaProperties properties;

    public GetuiRtaClient(GetuiRtaProperties properties) {
        super(properties);
        this.properties = properties;
    }

    @Override
    protected HttpRequestBase create(RtaRequest request) {
        HttpPost post = new HttpPost(properties.getUrl());

        String param = request.getParam();
        if (StringUtils.isBlank(param)) {
            throw new RtaRequestException("param");
        }
        String[] arr = StringUtils.splitPreserveAllTokens(param, "|");
        if (arr.length < 4) {
            throw new RtaRequestException("param must be {appKey}|{appSecret}|{targetType}|{adsType}");
        }
        String appKey = arr[0];
        String appSecret = arr[1];
        String targetType = arr[2];
        String adsType = arr[3];

        String timeStamp = TypeHelper.castToString(System.currentTimeMillis());
        post.addHeader("appKey", appKey);
        post.addHeader("timeStamp", timeStamp);
        post.addHeader("Content-Type", MediaType.APPLICATION_JSON_UTF8_VALUE);
        post.addHeader("token", DigestUtils.md5Hex(appKey + appSecret + timeStamp));

        GetuiRequest.Device dev = new GetuiRequest.Device()
                .setImei(request.getImei()).setImeiMd5(request.getImei_md5())
                .setOaid(request.getOaid()).setOaidMd5(request.getOaid_md5())
                .setIdfa(request.getIdfa()).setIdfaMd5(request.getIdfa_md5())
                .setAndroidId(request.getAndroidid()).setAndroidIdMd5(request.getAndroidid_md5());

        GetuiRequest.Condition condition = new GetuiRequest.Condition()
                .setAdsType(NumberUtils.toInt(adsType))
                .setTargetType(NumberUtils.toInt(targetType));

        GetuiRequest gt = new GetuiRequest()
                .setRequestId(UUIDHelper.generate())
                .setDevice(dev)
                .setQueryConditions(Lists.newArrayList(condition));

        post.setEntity(new StringEntity(JsonHelper.convert(gt), StandardCharsets.UTF_8));

        return post;
    }

    @Override
    protected boolean isTarget(RtaRequest request, String json) {
        GetuiResponse response = JsonHelper.convert(json, GetuiResponse.class);
        if (!response.isRequestOk()) {
            return false;
        }
        List<GetuiResponse.Data> data = response.getData();
        return CollectionUtils.isNotEmpty(data) && data.get(0).getResult() == 1;
    }

    @Override
    public RtaType getRtaType() {
        return RtaType.Getui;
    }
}
