package com.bxm.shop.integration;

import com.alibaba.fastjson.JSONObject;
import com.bxm.shop.common.exception.ResponseCodeType;
import com.bxm.shop.common.exception.ShopsException;
import com.bxm.shopmanager.facade.model.GoodsPoolDTO;
import com.bxm.shopmanager.facade.model.RebateDTO;
import com.bxm.shopmanager.facade.service.ConfigService;
import com.bxm.warcar.utils.response.ResultModel;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.dozer.Mapper;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;

/**
 * @author yirenjie
 * createDate:  2018/12/21
 */
@Service
@Slf4j
public class ShopManagerIntegration {

    @Resource
    private ConfigService configService;

    @Resource
    private Mapper mapper;

    public Map<Long, GoodsPoolDTO> getGoodsPoolsByIds(List<Long> ids) {
        try {
            if (ids == null) {
                throw new ShopsException(ResponseCodeType.PARAM_ILLEGAL, "商品池id不能为空");
            }
            ResultModel<Map<Long, GoodsPoolDTO>> response = configService.getGoodsPoolByIds(Lists.newArrayList(ids));
            if (response != null && response.isSuccessed()) {
                return response.getReturnValue();
            }
            log.error("请求管理后台系统获取商品池响应异常,请求参数{},响应参数:{}", JSONObject.toJSONString(ids), JSONObject.toJSONString(response));
        } catch (Exception e) {
            log.error("请求后台管理系统获取商品池发生异常:", e);
        }
        return Maps.newHashMap();
    }

    public GoodsPoolDTO getGoodsPoolById(Long id) {
        return getGoodsPoolsByIds(Lists.newArrayList(id)).get(id);
    }

    public Map<String,RebateConfig> getRebateByIds(List<String> ids) {
        Map<String, RebateConfig> result = Maps.newHashMap();
        try {
            if (ids == null || ids.isEmpty()) {
                throw new ShopsException(ResponseCodeType.PARAM_ILLEGAL, "商品id不能为空");
            }
            ResultModel<Map<String, RebateDTO>> response = configService.getRebateByIds(ids);
            if (response != null && response.isSuccessed()) {
                Map<String, RebateDTO> returnValue = response.getReturnValue();
                for (String key : returnValue.keySet()) {
                    result.put(key, mapper.map(returnValue.get(key), RebateConfig.class));
                }
                return result;
            }
            log.error("请求后台管理系统获取返佣返利配置响应异常,请求参数{},响应参数:{}", JSONObject.toJSONString(ids), JSONObject.toJSONString(response));
        }catch (Exception e) {
            log.error("请求后台管理系统获取返佣返利配置发生异常:", e);
        }
        return result;
    }

    public RebateConfig getRebateById(String id) {
        Map<String, RebateConfig> result = getRebateByIds(Lists.newArrayList(id));
        return result.get(id);
    }


    public enum GoodsPoolType {

        FREE(1L),

        TOP(2L);

        @Getter
        private Long id;

        GoodsPoolType(Long id) {
            this.id = id;
        }
    }

}
