package com.bxm.shop.integration.pdd.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.shop.common.exception.RedisConstants;
import com.bxm.shop.common.utils.OkHttpUtils;
import com.bxm.shop.common.utils.ParamEncodeUtils;
import com.bxm.shop.facade.model.goods.GoodsQueryRo;
import com.bxm.shop.integration.config.PddConstants;
import com.bxm.shop.integration.config.PingduoduoConfig;
import com.bxm.shop.integration.pdd.PddGoodsIntegration;
import com.bxm.shop.model.goods.vo.*;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @author yirenjie
 * createDate:  2018/11/8
 */
@Service
@Slf4j
public class PddGoodsIntegrationImpl implements PddGoodsIntegration {

    @Resource
    private PingduoduoConfig pingduoduoConfig;

    @Resource
    private StringRedisTemplate stringRedisTemplate;

    @Override
    public GoodsList.GoodsSearchResponse queryGoodsList(GoodsQueryRo goodsQueryRo) {
        String json = stringRedisTemplate.opsForValue().get(String.format(RedisConstants.PDD_GOODS_LIST_CACHE_KEY, goodsQueryRo.hashCode()));
        if (StringUtils.isNotBlank(json)) {
            return JSONObject.parseObject(json, GoodsList.GoodsSearchResponse.class);
        }
        Map<String, String> param = new HashMap<>();
        if (goodsQueryRo.getTagId() != null) {
            param.put("opt_id", goodsQueryRo.getTagId().toString());
        }
        if (StringUtils.isNotBlank(goodsQueryRo.getKeyword())) {
            param.put("keyword", goodsQueryRo.getKeyword());
        }
        if (goodsQueryRo.getGoodsIdList() != null) {
            param.put("goods_id_list", JSONObject.toJSONString(goodsQueryRo.getGoodsIdList()));
        }
        param.put("page_size", goodsQueryRo.getPageSize().toString());
        param.put("page", goodsQueryRo.getPageNum().toString());
        param.put("sort_type", goodsQueryRo.getSortType().toString());
        param.put("type", PddConstants.GOODS_SEARCH);
        param.put("with_coupon", "true");   // 固定只返回有优惠券的
        try {
            GoodsList goodsList = post(param, GoodsList.class);
            if (goodsList == null) {
                log.warn("查询拼多多获取商品列表接口返回空值");
                return new GoodsList.GoodsSearchResponse();
            }
            if (goodsList.getErrorResponse() != null) {
                log.warn("查询拼多多获取商品列表接口返回错误信息{},queryRo={}", goodsList, goodsQueryRo);
                return new GoodsList.GoodsSearchResponse();
            }
            GoodsList.GoodsSearchResponse result = goodsList.getResponse();
            // 设置缓存 5分钟
            stringRedisTemplate.opsForValue()
                    .set(String.format(RedisConstants.PDD_GOODS_LIST_CACHE_KEY, goodsQueryRo.hashCode())
                            , JSONObject.toJSONString(result), 5L, TimeUnit.MINUTES);
            return result;
        } catch (IOException e) {
            log.error("查询拼多多获取商品列表接口发生异常:", e);
        }
        return new GoodsList.GoodsSearchResponse();
    }

    @Override
    public List<Good> queryGoodsList(List<String> goodsIdList) {
        if (goodsIdList == null || goodsIdList.isEmpty()) {
            return Lists.newArrayList();
        }
        GoodsQueryRo ro = new GoodsQueryRo();
        ro.setGoodsIdList(goodsIdList);
        return queryGoodsList(ro).getGoodsList();
    }

    @Override
    public GoodDetail queryGoodDetail(String goodId) {
        String json = stringRedisTemplate.opsForValue().get(String.format(RedisConstants.PDD_GOODS_DETAIL_CACHE_KEY, goodId));
        if (StringUtils.isNotBlank(json)) {
            return JSONObject.parseObject(json, GoodDetail.class);
        }
        Map<String, String> param = new HashMap<>();
        String[] goodIds = {goodId};
        param.put("goods_id_list", JSONObject.toJSONString(goodIds));
        param.put("type", PddConstants.GOODS_DETAIL);
        try {
            GoodDetailList goodDetailList = post(param, GoodDetailList.class);
            if (goodDetailList != null && goodDetailList.getResponse() != null
                    && goodDetailList.getResponse().getGoodsDetails() != null
                    && goodDetailList.getResponse().getGoodsDetails().size() > 0) {

                GoodDetail result = goodDetailList.getResponse().getGoodsDetails().get(0);
                // 设置缓存 5分钟
                stringRedisTemplate.opsForValue()
                        .set(String.format(RedisConstants.PDD_GOODS_DETAIL_CACHE_KEY, goodId)
                                , JSONObject.toJSONString(result), 5L, TimeUnit.MINUTES);
                return result;
            }
        } catch (IOException e) {
            log.error("查询拼多多获取商品详情接口发生异常:", e);
        }
        return null;
    }

    @Override
    public List<Good> queryThemeGoodsList(Long themeId) {
        String redisKey = String.format(RedisConstants.PDD_THEME_GOODS_CACHE_KEY, themeId);
        String json = stringRedisTemplate.opsForValue().get(redisKey);
        if (StringUtils.isNotBlank(json)) {
            return JSONObject.parseArray(json, Good.class);
        }
        Map<String, String> param = new HashMap<>();
        param.put("theme_id", themeId.toString());
        param.put("type", PddConstants.THEME_GOODS);
        try {
            ThemeGoods response = post(param, ThemeGoods.class);
            if (response != null && response.getResponse() != null && response.getResponse().getGoodsList() != null) {
                List<Good> goodsList = response.getResponse().getGoodsList();
                // 设置缓存 5分钟
                stringRedisTemplate.opsForValue().set(redisKey, JSONObject.toJSONString(goodsList), 1L, TimeUnit.HOURS);
                return goodsList;
            }
        } catch (Exception e) {
            log.error("查询拼多多获取主题商品接口发生异常:", e);
        }
        return Lists.newArrayList();
    }

    private <T> T post(Map<String, String> param, Class<T> clazz) throws IOException {
        param.put("timestamp", String.valueOf(System.currentTimeMillis()).substring(0, 10));
        param.put("client_id",pingduoduoConfig.getClientId());
        String signature = ParamEncodeUtils.getMd5Signature(param, pingduoduoConfig.getClientSecret());
        param.put("sign", signature);
        String s = OkHttpUtils.post(PddConstants.URL, param);
        if (StringUtils.isNotBlank(s)) {
            s = ParamEncodeUtils.unicodeToString(s);
            return JSONObject.parseObject(s, clazz);
        }
        return null;
    }

}