package com.bxm.shop.service.impl;

import com.bxm.shop.common.exception.RedisConstants;
import com.bxm.shop.common.exception.ResponseCodeType;
import com.bxm.shop.common.exception.ShopsException;
import com.bxm.shop.config.WechatConfig;
import com.bxm.shop.dal.Boost;
import com.bxm.shop.dal.BoosterRecord;
import com.bxm.shop.dal.mapper.BoostMapper;
import com.bxm.shop.dal.mapper.BoosterRecordMapper;
import com.bxm.shop.facade.model.boost.BoostDetailVo;
import com.bxm.shop.facade.model.boost.CreateBoostRo;
import com.bxm.shop.facade.model.goods.FreeGoodsVo;
import com.bxm.shop.integration.RebateConfig;
import com.bxm.shop.integration.ShopManagerIntegration;
import com.bxm.shop.integration.pdd.PddGoodsIntegration;
import com.bxm.shop.model.goods.vo.Good;
import com.bxm.shop.model.user.dao.UserDao;
import com.bxm.shop.model.wechat.SendTemplateMessageRo;
import com.bxm.shop.service.BoostService;
import com.bxm.shop.service.GoodsService;
import com.bxm.shop.service.WechatService;
import com.bxm.shop.utils.RedisLock;
import com.bxm.shopmanager.facade.model.GoodsPoolDTO;
import com.bxm.warcar.cache.Fetcher;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.dozer.Mapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author yirenjie
 * createDate:  2018/12/21
 */
@Service
@Slf4j
public class BoostServiceImpl implements BoostService {

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;

    @Resource
    private StringRedisTemplate stringRedisTemplate;

    @Resource
    private BoostMapper boostMapper;

    @Resource
    private BoosterRecordMapper boosterRecordMapper;

    @Resource
    private PddGoodsIntegration pddGoodsIntegration;

    @Resource
    private GoodsService goodsService;

    @Resource
    private Mapper mapper;

    @Resource
    private WechatService wechatService;

    @Resource
    private ShopManagerIntegration shopManagerIntegration;

    @Resource
    private WechatConfig wechatConfig;

    @Resource
    private RedisLock redisLock;

    @Override
    public FreeGoodsVo list(String openid) {
        if (StringUtils.isBlank(openid)) {
            throw new ShopsException(ResponseCodeType.PARAM_ILLEGAL, "openid为空");
        }
        FreeGoodsVo freeGoodsVo = new FreeGoodsVo();
        GoodsPoolDTO goodsPoolDTO = shopManagerIntegration.getGoodsPoolById(ShopManagerIntegration.GoodsPoolType.FREE.getId());
        if (goodsPoolDTO == null) {
            log.error("查询0元购商品池返回null");
            throw new ShopsException(ResponseCodeType.SYSTEM_ERROR);
        }
        List<String> freeGoodsIds = goodsPoolDTO.getGoodsIds();

        List<Good> freeGoods = pddGoodsIntegration.queryGoodsList(freeGoodsIds);
        Map<String, RebateConfig> rebateConfigs = shopManagerIntegration.getRebateByIds(freeGoodsIds);
        freeGoodsVo.setFreeGoods(
                freeGoods.stream().map(good -> goodsService.trans2Vo(good, freeGoodsIds, rebateConfigs)).collect(Collectors.toList()));
        Boost boost = boostMapper.findLeastByOpenid(openid);
        if (boost == null) {
            freeGoodsVo.setPageType(FreeGoodsVo.PageType.UNBOOSTED);
            return freeGoodsVo;
        }
        freeGoodsVo.setPageType(trans2PageType(boost.getStatus()));
        freeGoodsVo.setBoostId(boost.getId());
        return freeGoodsVo;
    }

    /**
     * 获取详情
     *
     * @param openid 微信openid
     * @param boostId 助力id
     */
    @Override
    public BoostDetailVo getDetail(String openid, Long boostId) {
        Boost boost = boostMapper.selectByPrimaryKey(boostId);

        BoostDetailVo boostDetail = new BoostDetailVo();
        if (boost == null) {
            throw new ShopsException(ResponseCodeType.BOOST_NOT_EXIST);
        }
        // copy
        mapper.map(boost, boostDetail);
        boostDetail.setMine(boost.getOpenid().equals(openid)); // 是否为本人的助力页

        UserDao info = fetcher.hfetch(RedisConstants.User.getUserInfo(), boost.getOpenid(), UserDao.class);
        boostDetail.setAvatar(info.getAvatar());
        boostDetail.setNickname(info.getNickname());
        List<BoosterRecord> boosterRecords = boosterRecordMapper.queryListByBoostId(boost.getId());
        boostDetail.setBoosterAmount(boosterRecords.size());
        List<BoostDetailVo.Booster> boosters = Lists.newArrayList();
        for (BoosterRecord boosterRecord : boosterRecords) {
            if (boosterRecord.getBooster().equals(openid)) {
                boostDetail.setHasBoosted(true);
            }
            BoostDetailVo.Booster booster = new BoostDetailVo.Booster();
            UserDao boosterUser = fetcher.hfetch(RedisConstants.User.getUserInfo(), boosterRecord.getBooster(), UserDao.class);
            if (boosterUser == null) {
                log.error("参与boostId={}的助力用户openid={}在redis中为空", boost.getId(), boosterRecord.getBooster());
                continue;
            }
            booster.setAvatar(boosterUser.getAvatar());
            booster.setNickname(boosterUser.getNickname());
            boosters.add(booster);
        }
        boostDetail.setBoosters(boosters);
        boostDetail.setPageType(trans2PageType(boost.getStatus()));
        return boostDetail;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void boost(String openid, Long boostId) {
        String lockKey = "BOOST_" + boostId;
        try {
            redisLock.lock(lockKey);
            if (StringUtils.isBlank(openid) || boostId == null) {
                throw new ShopsException(ResponseCodeType.PARAM_ILLEGAL);
            }
            Boost boost = boostMapper.selectByPrimaryKey(boostId);
            if (boost == null) {
                throw new ShopsException(ResponseCodeType.BOOST_NOT_EXIST);
            }
            if (!boost.getStatus().isBoosting()) {
                throw new ShopsException(ResponseCodeType.BOOST_HAS_FINISHED);
            }
            if (boost.getOpenid().equals(openid)) {
                throw new ShopsException(ResponseCodeType.BOOST_SELF_ILLEGAL);
            }
            if (boosterRecordMapper.existByOpenidAndBoostId(boostId, openid)) {
                return;
            }
            BoosterRecord boosterRecord = new BoosterRecord();
            boosterRecord.setBooster(openid);
            boosterRecord.setBoostId(boostId);
            if (boosterRecordMapper.insertSelective(boosterRecord) != 1) {
                throw new ShopsException(ResponseCodeType.SYSTEM_ERROR);
            }
            Integer boostCount = boosterRecordMapper.countByBoostId(boostId);
            if (boostCount < boost.getRequiredAmount()) {
                return;
            }

            // 助力完成逻辑
            if (!boostMapper.updateStatusByBoostId(boostId, Boost.Status.S.name())) {
                throw new ShopsException(ResponseCodeType.SYSTEM_ERROR);
            }
            SendTemplateMessageRo ro = new SendTemplateMessageRo();
            ro.setToUser(boost.getOpenid());
            ro.setTemplateId(wechatConfig.getBoostTemplateId());
            ro.setPage("/pages/free/free?id=" + boostId);
            ro.setFormId(boost.getFormId());
            SendTemplateMessageRo.Data data = SendTemplateMessageRo.Data.builder()
                    .keyword1(SendTemplateMessageRo.Value.builder().value("您的助力已经完成，赶紧享受0元购机会").build())
                    .keyword2(SendTemplateMessageRo.Value.builder().value("点击该消息即可跳转购买").build()).build();
            ro.setData(data);
            wechatService.sendTemplateMessage(ro);
        } finally {
            redisLock.unlock(lockKey);
        }

    }

    @Override
    public Long createBoost(CreateBoostRo createBoostRo) {
        String lockKey = "CREATE_BOOST_" + createBoostRo.getOpenid();
        try {
            redisLock.lock(lockKey);
            Integer count = boostMapper.countByOpenid(createBoostRo.getOpenid());
            int freeTimes = Integer.valueOf(stringRedisTemplate.opsForValue().get(RedisConstants.FREE_TIMES));
            if (count >= freeTimes) {
                throw new ShopsException(ResponseCodeType.BOOST_HAS_NO_CHANCE);
            }
            Boost boost = mapper.map(createBoostRo, Boost.class);
            if (boostMapper.insertSelective(boost) != 1) {
                throw new ShopsException(ResponseCodeType.SYSTEM_ERROR);
            }
            return boost.getId();
        } finally {
            redisLock.unlock(lockKey);
        }
    }

    @Override
    public String boostStatus(String openid) {
        Boost boost = boostMapper.findLeastByOpenid(openid);
        if (boost == null || boost.getStatus().isBoosting()) {
            return FreeGoodsVo.PageType.BOOSTING.name();
        }
        return boost.getStatus().isFinished() ? FreeGoodsVo.PageType.USED.name() : FreeGoodsVo.PageType.BOOSTED.name();
    }

    private FreeGoodsVo.PageType trans2PageType(Boost.Status status) {

        switch (status) {
            case S:
                return FreeGoodsVo.PageType.BOOSTED;
            case I:
            case H:
                return FreeGoodsVo.PageType.BOOSTING;
            case U:
                return FreeGoodsVo.PageType.USED;
        }
        return null;
    }

}
