package com.bxm.shop.service.impl;

import com.bxm.shop.common.enums.WithdrawStatusEnum;
import com.bxm.shop.common.exception.RedisConstants;
import com.bxm.shop.common.exception.ResponseCodeType;
import com.bxm.shop.common.utils.ExceptionPrintUtils;
import com.bxm.shop.config.BizConfig;
import com.bxm.shop.config.WechatConfig;
import com.bxm.shop.dal.mapper.UserProfitMapper;
import com.bxm.shop.dal.mapper.UserWithdrawMapper;
import com.bxm.shop.dal.mapper.WechatWithdrawRecordMapper;
import com.bxm.shop.facade.model.wallet.WalletDto;
import com.bxm.shop.facade.model.wallet.WalletVo;
import com.bxm.shop.integration.config.PingduoduoConfig;
import com.bxm.shop.model.wallet.dao.UserWithdrawDao;
import com.bxm.shop.model.wallet.dao.WalletRedisDao;
import com.bxm.shop.model.wechat.WechatWithdrawRecordDao;
import com.bxm.shop.service.WalletSerive;
import com.bxm.shop.utils.DistributedLock;
import com.bxm.shop.wxpay.WxPayUtil;
import com.bxm.shop.wxpay.dto.PayPerson;
import com.bxm.shop.wxpay.dto.PayPersonReturn;
import com.bxm.shop.wxpay.dto.TransferinfoDto;
import com.bxm.shop.wxpay.dto.TransferinfoReturn;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.message.Message;
import com.bxm.warcar.message.MessageSender;
import com.bxm.warcar.utils.response.ResultModel;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.dozer.Mapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

@Service
@Slf4j
public class WalletSeriveImpl implements WalletSerive {

    @Autowired
    private WechatConfig wechatConfig;

    @Autowired
    private WechatWithdrawRecordMapper wechatWithdrawRecordMapper;

    @Autowired
    private Mapper mapper;

    @Autowired
    private MessageSender messageSender;

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;

    @Autowired
    @Qualifier("jedisUpdater")
    protected Updater updater;

    @Autowired
    private UserProfitMapper userProfitMapper;

    @Autowired
    private UserWithdrawMapper userWithdrawMapper;

    @Autowired
    private PingduoduoConfig pingduoduoConfig;

    @Autowired
    private BizConfig bizConfig;

    @Autowired
    private DistributedLock distributedLock;

    private ThreadPoolExecutor poolExecutor = new ThreadPoolExecutor(Runtime.getRuntime().availableProcessors()*2,
            Runtime.getRuntime().availableProcessors()*2,0L, TimeUnit.SECONDS,new LinkedBlockingDeque(),
            new ThreadFactoryBuilder().setNameFormat("wallet-deal-pool-%d").build()
    );
    @Override
    public ResultModel withdraw(WalletDto dto) {
        ResultModel rs = new ResultModel();
        PayPerson payPersonBean = null;
        PayPersonReturn payPersonReturn = null;
        String indentifier = null;
        try {
            //验证参数是否正确
            if(StringUtils.isBlank(dto.getOpenid()) || dto.getAmount()<=0 || StringUtils.isBlank(dto.getToken())){
                rs.setSuccessed(Boolean.FALSE);
                rs.setErrorCode(ResponseCodeType.PARAM_ILLEGAL.getErrorCode());
                rs.setErrorDesc(ResponseCodeType.PARAM_ILLEGAL.getErrorMsg());
                rs.setReturnValue(ResponseCodeType.PARAM_ILLEGAL.getErrorMsg());
                return rs;
            }
            indentifier = distributedLock.lockWithTimeout(RedisConstants.Wallet.getWithdrawLongck(dto.getOpenid()).generateKey(),3000,3000);
            //验证参数
            if (checkParam(dto, rs)){
                return rs;
            }
            //生成提现记录
            UserWithdrawDao withdrawInfo = new UserWithdrawDao();
            withdrawInfo.setAmount(dto.getAmount());
            withdrawInfo.setOpenid(dto.getOpenid());
            withdrawInfo.setCreateTime(new Date());
            withdrawInfo.setPayStatus(WithdrawStatusEnum.FREEZE.getCode());
            try {
                withdrawInfo.setNonceStr(WxPayUtil.createUniqueCode(20));
            } catch (Exception e) {
                log.error(" NoceStr createUniqueCode Error ",e);
            }
            userWithdrawMapper.save(withdrawInfo);
            //刷新redis缓存,提现或者冻结金额
            refreshFreezeOrUsedAmount(dto.getOpenid());

            //商户订单号,确保唯一性,提现表id加订单生成时间
            String partnerTradeNo = genPartnerTradeNo(withdrawInfo.getId());
            withdrawInfo.setPartnerTradeNo(partnerTradeNo);
            payPersonBean = getBasicPayPerson();
            payPersonBean.setNonceStr(withdrawInfo.getNonceStr());
            payPersonBean.setPartnerTradeNo(partnerTradeNo);
            payPersonBean.setOpenid(withdrawInfo.getOpenid());
            payPersonBean.setAmount(withdrawInfo.getAmount().intValue());
//            payPersonBean.setAmount(100);
            try {
                payPersonReturn = WxPayUtil.payPersion(payPersonBean, wechatConfig);

            } catch (Exception e) {
                Message msg = new Message();
                msg.setContent("withdraw  error :" + ExceptionPrintUtils.printStackTraceToString(e));
                messageSender.send2(msg);
                log.error("withdraw  error",e);
            }finally {
                if(null != payPersonReturn && "SUCCESS".equals(payPersonReturn.getResultCode())) {
                    withdrawInfo.setModifyTime(new Date());
                    withdrawInfo.setPaymentNo(payPersonReturn.getPaymentNo());
                    withdrawInfo.setPaymentTime(payPersonReturn.getPaymentTime());
                    withdrawInfo.setPayStatus(WithdrawStatusEnum.SUCCESS.getCode());
                    userWithdrawMapper.update(withdrawInfo);
                    //更新提现表状态
                    rs.setReturnValue("提现成功");
                }else{
                    //出错的话重试
                    rs.setSuccessed(Boolean.FALSE);
                    rs.setErrorCode(ResponseCodeType.USER_CASHBACKING.getErrorCode());
                    rs.setErrorDesc(ResponseCodeType.USER_CASHBACKING.getErrorMsg());
                    poolExecutor.submit(new RetryWechatPay(payPersonBean,withdrawInfo));
                }
                return rs;
            }
        }finally {
            //不管接口调用是否成功都保存接口调用记录
            poolExecutor.submit( new WechatWithdrawRecordSaveThread(payPersonBean,payPersonReturn) );
            distributedLock.releaseLock(RedisConstants.Wallet.getWithdrawLongck(dto.getOpenid()).generateKey(),indentifier);
        }
    }

    /**
     * 刷新已提现金额（包括冻结金额）缓存
     * @param openid
     */
    private void refreshFreezeOrUsedAmount(String openid) {
        //计算已提现金额或者冻结金额
        Long haveWithdrawalAmount = userWithdrawMapper.haveWithdrawalAmount(openid);
        if(null == haveWithdrawalAmount){
            haveWithdrawalAmount = 0L;
        }
        updater.hupdate(RedisConstants.Wallet.getFreezeOrUsed(),openid,haveWithdrawalAmount);
    }

    /**
     * 生成商户订单号
     * @param id
     * @return
     */
    private String genPartnerTradeNo(Long id) {
        try {
            return String.format("%014d", id)+WxPayUtil.createUniqueCode(16);
        } catch (Exception e) {
            log.error(" partnerTradeNo createUniqueCode Error ",e);
            return String.format("%014d", id);
        }
    }

    private boolean checkParam(WalletDto dto, ResultModel rs) {

        //提现金额阈值
        if(dto.getAmount()< bizConfig.getAmountLeastThreshold()){
            rs.setSuccessed(Boolean.FALSE);
            rs.setErrorCode(ResponseCodeType.AMOUNT_LESS_THAN_THRESHOLD.getErrorCode());
            rs.setErrorDesc(ResponseCodeType.AMOUNT_LESS_THAN_THRESHOLD.getErrorMsg());
            rs.setReturnValue(ResponseCodeType.AMOUNT_LESS_THAN_THRESHOLD.getErrorMsg());
            return true;
        }
        //验证token
        String token = fetcher.fetch(RedisConstants.Wallet.getWalletToken(dto.getOpenid()), String.class);
        if(StringUtils.isEmpty(token) || !token.equals(dto.getToken())){
            rs.setSuccessed(Boolean.FALSE);
            rs.setErrorCode(ResponseCodeType.PARAM_ILLEGAL.getErrorCode());
            rs.setErrorDesc(ResponseCodeType.PARAM_ILLEGAL.getErrorMsg());
            rs.setReturnValue(ResponseCodeType.PARAM_ILLEGAL.getErrorMsg());
            return true;
        }
        //验证可提现金额是否正确
        Long withdrawAmount = getWithdrawAmount(dto.getOpenid());
        if(dto.getAmount()>withdrawAmount ){
            rs.setSuccessed(Boolean.FALSE);
            rs.setErrorCode(ResponseCodeType.AMOUNT_NOT_ENOUGH.getErrorCode());
            rs.setErrorDesc(ResponseCodeType.AMOUNT_NOT_ENOUGH.getErrorMsg());
            rs.setReturnValue(ResponseCodeType.AMOUNT_NOT_ENOUGH.getErrorMsg());
            return true;
        }
        return false;
    }


    private PayPerson getBasicPayPerson(){
        PayPerson payPersonBean = new PayPerson(wechatConfig.getAppId(), wechatConfig.getMchId());
        //不验证提现人真实姓名
        payPersonBean.setCheckName("NO_CHECK");
        payPersonBean.setDesc(bizConfig.getPayDesc());
        try {
            InetAddress address = InetAddress.getLocalHost();
            payPersonBean.setSpbillCreateIp(address.getHostAddress());
        } catch (UnknownHostException e) {
            log.error("获取本机ip出错",e);
            payPersonBean.setSpbillCreateIp("192.168.0.1");
        }
        return payPersonBean;
    }

    @Override
    public WalletVo info(WalletDto dto) {
        WalletVo vo = new WalletVo();
        vo.setToken(UUID.randomUUID().toString());
        //提现阀值
        vo.setAmountLeastThreshold(bizConfig.getAmountLeastThreshold());

        WalletRedisDao cache = fetcher.hfetch(RedisConstants.Wallet.getWalletInfo(), dto.getOpenid(), WalletRedisDao.class);
        if(null !=cache){
            vo.setContributeAmount(cache.getContributeAmount());
            vo.setWithdrawAmount(cache.getWithdrawAmount());
            vo.setEstimateAmount(cache.getEstimateAmount());
            vo.setRebateAmount(cache.getRebateAmount());
        }else {
            vo.setContributeAmount(0L);
            vo.setWithdrawAmount(0L);
            vo.setEstimateAmount(0L);
            vo.setRebateAmount(0L);
        }

        Long haveWithdrawalAmount = fetcher.hfetch(RedisConstants.Wallet.getFreezeOrUsed(),dto.getOpenid(),Long.class);
        if(null == haveWithdrawalAmount){
            haveWithdrawalAmount = 0L;
        }
        vo.setWithdrawAmount(vo.getWithdrawAmount()-haveWithdrawalAmount);

        //设置提现token
        updater.update(RedisConstants.Wallet.getWalletToken(dto.getOpenid()),vo.getToken(),RedisConstants.User.SESSION_TIME);
        return vo;
    }

    @Override
    public void cacheProfit(String openid) {
        WalletRedisDao dao = new WalletRedisDao();
        //好友贡献金额
        Long freidnContibute = userProfitMapper.friendContributeAmount(openid, pingduoduoConfig.getSold());
        if(freidnContibute == null){
            freidnContibute = 0L;
        }
        dao.setContributeAmount(freidnContibute);
        //获取预估总额
        Long estimateAmount = userProfitMapper.getAmountByOpenidAndStates(openid, pingduoduoConfig.getSold());
        if(null == estimateAmount){
            estimateAmount = 0L;
        }
        dao.setEstimateAmount(estimateAmount);
        //可提现金额
        Long withdrawAmount = userProfitMapper.getAmountByOpenidAndStates(openid, pingduoduoConfig.getCashbacked());
        if(withdrawAmount == null){
            withdrawAmount = 0L;
        }
        dao.setWithdrawAmount(withdrawAmount);
        //待返现总额
        Long rebateAmount = userProfitMapper.getAmountByOpenidAndStates(openid, pingduoduoConfig.getCashbacking());
        if(null == rebateAmount){
            rebateAmount = 0L;
        }
        dao.setRebateAmount(rebateAmount);
        updater.hupdate(RedisConstants.Wallet.getWalletInfo(),openid,dao);
    }

    @Override
    public void payRetry() throws Exception {
        //增加开关配置，是否使用支持重试功能
        String swtich = fetcher.fetch(RedisConstants.Wallet.getRetrySwitch(), String.class);
        if(null != swtich && "false".equals(swtich)){
            return ;
        }
        List<UserWithdrawDao> failList = userWithdrawMapper.findFailList();
        if(CollectionUtils.isNotEmpty(failList)){
            for(UserWithdrawDao dao : failList){
                //先查询是否提现成功
                TransferinfoDto transferinfoDto = new TransferinfoDto();
                transferinfoDto.setAppid(wechatConfig.getAppId());
                transferinfoDto.setMch_id(wechatConfig.getMchId());
                transferinfoDto.setPartner_trade_no(dao.getPartnerTradeNo());
                TransferinfoReturn transferinfoReturn = WxPayUtil.payQuery(transferinfoDto, wechatConfig);
                if(null != transferinfoReturn && "SUCCESS".equals(transferinfoReturn.getResultCode())){
                    dao.setPayStatus(WithdrawStatusEnum.SUCCESS.getCode());
                    dao.setErrCodeDes("null");
                    dao.setErrCode("null");
                    dao.setPaymentNo(transferinfoReturn.getDetailId());
                    dao.setPaymentTime(transferinfoReturn.getTransferTime());
                    dao.setModifyTime(new Date());
                    userWithdrawMapper.update(dao);
                    continue;
                }
                //重试支付接口
                PayPerson payPersonBean = getBasicPayPerson();
                payPersonBean.setPartnerTradeNo(dao.getPartnerTradeNo());
                payPersonBean.setNonceStr(dao.getNonceStr());
                payPersonBean.setOpenid(dao.getOpenid());
                payPersonBean.setAmount(dao.getAmount().intValue());
                PayPersonReturn payPersonReturn = null;
                try {
                     payPersonReturn = WxPayUtil.payPersion(payPersonBean, wechatConfig);
                    if(null != payPersonBean && !"SUCCESS".equals(payPersonReturn.getResultCode())){
                        log.error("payRetry error");
                        Message msg = new Message();
                        msg.setContent("payRetry  error : code"+payPersonReturn.getErrCode()+"  msg:  "+payPersonReturn.getErrCodeDes()
                                +" partnerTradeNo : "+dao.getPartnerTradeNo() );
                        messageSender.send2(msg);
                    }else if(null != payPersonBean && "SUCCESS".equals(payPersonReturn.getResultCode())){
                        dao.setPayStatus(WithdrawStatusEnum.SUCCESS.getCode());
                        dao.setErrCodeDes("null");
                        dao.setErrCode("null");
                        dao.setPaymentNo(payPersonReturn.getPaymentNo());
                        dao.setPaymentTime(payPersonReturn.getPaymentTime());
                        dao.setModifyTime(new Date());
                        userWithdrawMapper.update(dao);
                        Message msg = new Message();
                        msg.setContent("payRetry  success : PaymentNo"+payPersonReturn.getPaymentNo()+
                                " partnerTradeNo : "+dao.getPartnerTradeNo() );
                        messageSender.send2(msg);
                    }
                } catch (Exception e) {
                    log.error("payRetry error");
                    Message msg = new Message();
                    msg.setContent("payRetry  error :" + ExceptionPrintUtils.printStackTraceToString(e));
                    messageSender.send2(msg);
                }finally {
                    //不管接口调用是否成功都保存接口调用记录
                    poolExecutor.submit( new WechatWithdrawRecordSaveThread(payPersonBean,payPersonReturn) );
                }
            }
        }
    }

    /**
     * 获取可提现金额
     * @param openid
     * @return
     */
    private Long getWithdrawAmount(String openid){
        //可提现金额
        WalletRedisDao cache = fetcher.hfetch(RedisConstants.Wallet.getWalletInfo(), openid, WalletRedisDao.class);
        Long withdrawAmount = 0L;
        if(null == cache){
            return 0L;
        }else{
            withdrawAmount = cache.getWithdrawAmount();
        }
        //计算已提现金额或者冻结金额
        Long haveWithdrawalAmount = userWithdrawMapper.haveWithdrawalAmount(openid);
        if(null == haveWithdrawalAmount ){
            return withdrawAmount;
        }else {
            return withdrawAmount - haveWithdrawalAmount;
        }
    }

    /**
     * 重试提现接口
     */
    class RetryWechatPay extends Thread{
        private PayPerson payPerson;
        private UserWithdrawDao withdrawInfo;

        public RetryWechatPay(PayPerson payPerson,UserWithdrawDao withdrawInfo){
            this.payPerson = payPerson;
            this.withdrawInfo = withdrawInfo;
        }
        public PayPerson getPayPerson() {
            return payPerson;
        }

        public void setPayPerson(PayPerson payPerson) {
            this.payPerson = payPerson;
        }

        public UserWithdrawDao getWithdrawInfo() {
            return withdrawInfo;
        }

        public void setWithdrawInfo(UserWithdrawDao withdrawInfo) {
            this.withdrawInfo = withdrawInfo;
        }

        @Override
        public void run() {
            int count = 0;
            try {
                //1秒后重试
                Thread.sleep(1000);
            } catch (InterruptedException e) {

            }
            do {
                PayPersonReturn payPersonReturn = null;
                try {
                    try {
                        payPersonReturn = WxPayUtil.payPersion(payPerson, wechatConfig);
                    }finally {
                        //不管接口调用是否成功都保存接口调用记录
                        poolExecutor.submit( new WechatWithdrawRecordSaveThread(payPerson,payPersonReturn) );
                    }
                    if(null != payPersonReturn && "SUCCESS".equals(payPersonReturn.getResultCode())) {
                        withdrawInfo.setModifyTime(new Date());
                        withdrawInfo.setPaymentNo(payPersonReturn.getPaymentNo());
                        withdrawInfo.setPaymentTime(payPersonReturn.getPaymentTime());
                        withdrawInfo.setPayStatus(WithdrawStatusEnum.SUCCESS.getCode());
                        withdrawInfo.setModifyTime(new Date());
                        userWithdrawMapper.update(withdrawInfo);
                        return ;
                    } else {
                        //失败3次保存失败记录，发钉钉
                        if(count == 2) {
                            withdrawInfo.setErrCode(payPersonReturn.getErrCode());
                            withdrawInfo.setErrCodeDes(payPersonReturn.getErrCodeDes());
                            withdrawInfo.setPayStatus(WithdrawStatusEnum.FAIL.getCode());
                            withdrawInfo.setModifyTime(new Date());
                            userWithdrawMapper.update(withdrawInfo);
                            Message msg = new Message();
                            msg.setContent("RetryWechatPay pay error PartnerTradeNo:"+withdrawInfo.getPartnerTradeNo());
                            messageSender.send2(msg);
                            return;
                        }
                    }
                    Thread.sleep(1000);
                } catch (Exception e) {
                    log.error("RetryWechatPay error",e);
                }finally {
                    count++;
                }

            }while (count<3);

        }
    }

    /**
     * 异步保存微信提现记录
     */
    class WechatWithdrawRecordSaveThread extends Thread{
        private PayPerson payPerson;
        private PayPersonReturn payPersonReturn;
        public WechatWithdrawRecordSaveThread(PayPerson payPerson,PayPersonReturn payPersonReturn){
            this.payPerson = payPerson;
            this.payPersonReturn = payPersonReturn;
        }
        @Override
        public void run() {
            try {
                WechatWithdrawRecordDao wechatWithdrawRecordDao = new WechatWithdrawRecordDao();
                if (null != payPerson) {
                    mapper.map(payPerson, wechatWithdrawRecordDao);
                }
                if (null != payPersonReturn) {
                    mapper.map(payPersonReturn, wechatWithdrawRecordDao);
                }
                wechatWithdrawRecordDao.setCreateTime(new Date());
                wechatWithdrawRecordMapper.save(wechatWithdrawRecordDao);
            }catch (Exception e){
                Message msg = new Message();
                msg.setContent("WechatWithdrawRecordDao save error");
                messageSender.send2(msg);
                log.error("WechatWithdrawRecordDao save error",e);
            }
        }

    }
}
