package com.bxm.shopping.integration.adsmedia;

import com.bxm.adsmedia.facade.model.appentrance.AppEntranceAdRO;
import com.bxm.adsmedia.facade.model.appentrance.AppEntranceUrlVO;
import com.bxm.adsmedia.facade.model.appentrance.QueryAppEntranceParamDTO;
import com.bxm.util.dto.ResultModel;
import com.github.pagehelper.PageInfo;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;

import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * @author tangx
 * @date 2021/5/26 14:34
 */
@Slf4j
@Configuration
public class AdsmediaIntegration {

    @Autowired
    private AdsmediaFeignService adsmediaFeignService;

    /**
     * 媒体和广告位代号格式
     */
    public static final String CODE_NAME_FORMAT = "%06d";

    public AppEntranceAdRO findByPositionId(String positionId) {
        try {
            ResultModel<AppEntranceAdRO> resultModel = adsmediaFeignService.findByPositionId(positionId);
            if (resultModel.isSuccessed()) {
                return resultModel.getReturnValue();
            }
            log.error("开发者后台异常，根据positionId查询广告位失败！-->{},{}", resultModel.getErrorCode(), resultModel.getErrorDesc());
        } catch (Exception e) {
            log.error("根据positionId查询广告位异常", e);
        }
        return null;
    }

    public AppEntranceUrlVO getUrl(String positionId) {
        try {
            ResultModel<AppEntranceUrlVO> resultModel = adsmediaFeignService.getUrl(positionId);
            if (resultModel.isSuccessed()) {
                return resultModel.getReturnValue();
            }
            log.error("开发者后台异常，获取广告位链接失败！-->{},{}", resultModel.getErrorCode(), resultModel.getErrorDesc());
        } catch (Exception e) {
            log.error("获取广告位链接异常", e);
        }
        return null;
    }

    /**
     * Description: 获取广告位所有代号
     * JDK version used:<JDK1.8>
     */
    public Map<String, String> getAppEntranceCodeName() {
        List<AppEntranceAdRO> appEntranceAdROS = this.getAllList(new QueryAppEntranceParamDTO());
        if (CollectionUtils.isEmpty(appEntranceAdROS)) {
            return Collections.emptyMap();
        }
        Map<String, String> map = Maps.newHashMapWithExpectedSize(appEntranceAdROS.size());
        for (AppEntranceAdRO appEntranceAdRO : appEntranceAdROS) {
            map.put(String.format(CODE_NAME_FORMAT, appEntranceAdRO.getCode()) + "-" + appEntranceAdRO.getAppEntranceId(), appEntranceAdRO.getPositionId());
        }
        return map;
    }

    /**
     * Description: 从开发者后台调用查询广告位 全部
     * JDK version used:<JDK1.8>
     */
    public List<AppEntranceAdRO> getAllList(QueryAppEntranceParamDTO dto) {
        dto.setPageNum(dto.getPageNum());
        dto.setPageSize(Integer.MAX_VALUE);
        return getPage(dto).getList();
    }

    /**
     * Description: 从开发者后台调用查询广告位 分页
     * JDK version used:<JDK1.8>
     */
    public PageInfo<AppEntranceAdRO> getPage(QueryAppEntranceParamDTO dto) {
        try {
            ResultModel<PageInfo<AppEntranceAdRO>> resultModel = adsmediaFeignService.getPage(dto);
            if (resultModel.isSuccessed()) {
                return resultModel.getReturnValue();
            }
            log.error("开发者后台异常，查询广告位列表失败！-->{},{}", resultModel.getErrorCode(), resultModel.getErrorDesc());
        } catch (Exception e) {
            log.error("查询开发者后台的广告位列表异常", e);
        }
        return noneDataPage();
    }

    /**
     * Description: 返回一个空结果的分页对象
     * JDK version used:<JDK1.8>
     */
    public static <T> PageInfo<T> noneDataPage() {
        PageInfo<T> page = new PageInfo<>();
        page.setPageNum(1);
        page.setPageSize(20);
        page.setTotal(0L);
        page.setSize(0);
        page.setPages(0);
        page.setStartRow(0);
        page.setEndRow(0);
        page.setList(Collections.emptyList());
        page.setPrePage(0);
        page.setNextPage(0);
        page.setIsFirstPage(true);
        page.setIsLastPage(true);
        page.setHasPreviousPage(false);
        page.setHasNextPage(false);
        return page;
    }
}
