package com.bxm.shopping.web.controller;

import com.bxm.shopping.common.enums.MarkStrategyEnum;
import com.bxm.shopping.common.enums.OrderType;
import com.bxm.shopping.common.exception.ShoppingException;
import com.bxm.shopping.common.utils.DateUtil;
import com.bxm.shopping.common.utils.EasyExcelUtil;
import com.bxm.shopping.common.utils.RegularUtils;
import com.bxm.shopping.model.query.UserOrderQuery;
import com.bxm.shopping.model.vo.ExprortOrderVo;
import com.bxm.shopping.model.vo.UserOrderVo;
import com.bxm.shopping.service.IUserOrderService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * @author 拉拉
 * @version v1.0.0
 * @created 2020-07-14 20:42:00
 * @modifier 拉拉
 * @updated 2020-07-14 20:42:00
 */
@Slf4j
@RestController
public class OrderController {

    @Resource
    IUserOrderService orderService;


    @RequestMapping(value = "/order/getPage", produces = "application/json", method = RequestMethod.GET)
    public ResponseEntity getPage(UserOrderQuery orderQuery) {
        orderQuery.setPlatform(UserOrderQuery.PLATFORM_OP);
        return ResponseEntity.ok(orderService.getPage(orderQuery));
    }

    @RequestMapping(value = "/order/updateStatus", produces = "application/json", method = RequestMethod.GET)
    public ResponseEntity updateStatus(@RequestParam(value = "id") Integer id,
                                  @RequestParam(value = "status") Integer status) {
        orderService.updateStatus(id, status);
        return ResponseEntity.ok("true");
    }

    /**
     * 导出订单
     * 由于导出数据量有时过大30w+,重复点击导致内存占满。故用synchronized处理
     * @param response
     * @param orderQuery
     */
    @RequestMapping(value = "/order/export", method = RequestMethod.GET)
    public synchronized void export(HttpServletResponse response,
                       UserOrderQuery orderQuery) {
        orderQuery.setPageNum(1);
        orderQuery.setPageSize(Integer.MAX_VALUE);
        orderQuery.setPlatform(UserOrderQuery.PLATFORM_OP);
        List<UserOrderVo> list = orderService.getPage(orderQuery).getRecords();
        String fileName = DateUtil.dateTo81String(new Date()) + "订单";

        List<ExprortOrderVo> exprortOrderVosList = new ArrayList<>();
        for (UserOrderVo orderVo : list) {
            ExprortOrderVo exprortOrderVo = new ExprortOrderVo();
            BeanUtils.copyProperties(orderVo, exprortOrderVo);
            exprortOrderVo.setOrderStatus(OrderType.getByCode(orderVo.getStatus()).getDesc());
            exprortOrderVo.setUserMobile(RegularUtils.encryptPhone(exprortOrderVo.getUserMobile()));
            exprortOrderVo.setProductStyle(StringUtils.replace(orderVo.getProductStyle(), "&", " \r\n"));
            // 自建站后台不显示身份证
            exprortOrderVo.setIdCard("");
            exprortOrderVo.setPayTypeName(orderVo.getPayType() == null ? null : (orderVo.getPayType() == 1 ? "支付宝" : "微信") );
            exprortOrderVo.setPayStatusName(orderVo.getPayStatus() == null ? null : (orderVo.getPayStatus() == 0 ? "未支付" : "已支付") );
            String hitMarks = orderVo.getHitMarks();

            if (StringUtils.isNotBlank(hitMarks)) {
                StringBuilder sb = new StringBuilder();
                String[] marks = hitMarks.split(",");
                for (String mark : marks) {
                    sb.append(MarkStrategyEnum.getNameByMarkType(Integer.parseInt(mark)));
                }
                exprortOrderVo.setHitMarks(sb.toString());
            }
            exprortOrderVo.setHitRiskWords(orderVo.getHitRiskWords());

            if (null != orderVo.getBehaviorType()) {
                Integer behaviorType = orderVo.getBehaviorType();
                switch (behaviorType) {
                    case 1: exprortOrderVo.setBehaviorType("入库");
                    break;
                    case 2: exprortOrderVo.setBehaviorType("激活");
                    break;
                    case 3: exprortOrderVo.setBehaviorType("充值");
                    break;
                    case 4: exprortOrderVo.setBehaviorType("恶意订单");
                    break;
                    default: exprortOrderVo.setBehaviorType(behaviorType.toString());
                }
            }

            exprortOrderVosList.add(exprortOrderVo);
        }

        //开始导出
        EasyExcelUtil.write(response, fileName, ExprortOrderVo.class, exprortOrderVosList);
    }

    @RequestMapping(value = "/order/importOrder", method = RequestMethod.POST)
    public synchronized ResponseEntity importOrder(@RequestParam("file") MultipartFile file, @RequestParam("company") String company) {

        List<ExprortOrderVo> orderVos = null;
        try {
            orderVos = EasyExcelUtil.read(file.getInputStream(), ExprortOrderVo.class);
        } catch (IOException e) {
            throw new ShoppingException("文件解析异常，请联系管理员");
        }
        if (CollectionUtils.isEmpty(orderVos)) {
            throw new ShoppingException("不能导入空文件");
        }
        log.info(orderVos.toString());
        return ResponseEntity.ok(orderService.importOrder(orderVos, company, null));
    }

    @RequestMapping(value = "/order/getOrderInfo", method = RequestMethod.GET)
    public ResponseEntity getOrderInfo(Integer id) {
        return ResponseEntity.ok(orderService.getOrderInfo(id));
    }

    /**
     * 批量取消发货
     * @param file
     * @return
     */
    @RequestMapping(value = "/order/batchCancel", method = RequestMethod.POST)
    public ResponseEntity batchCancel(@RequestParam("file") MultipartFile file) {

        List<ExprortOrderVo> orderVos = null;
        try {
            orderVos = EasyExcelUtil.read(file.getInputStream(), ExprortOrderVo.class);
        } catch (IOException e) {
            throw new ShoppingException("文件解析异常，请联系管理员");
        }
        if (CollectionUtils.isEmpty(orderVos)) {
            throw new ShoppingException("不能导入空文件");
        }
        return ResponseEntity.ok(orderService.batchCancel(orderVos, null));
    }

    /**
     * 单个订单发货
     * @return
     */
    @RequestMapping(value = "/order/updateSendStatus", method = RequestMethod.POST)
    public ResponseEntity updateSendStatus(@RequestParam("id") Integer id,
                                           @RequestParam("trackingNum") String trackingNum,
                                           @RequestParam("company") String company) {
        orderService.updateSendStatus(id, trackingNum, company);
        return ResponseEntity.ok(true);
    }



}
