package com.bxm.shopping.web.controller;


import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.shopping.common.enums.TemplateTypeEnum;
import com.bxm.shopping.common.redisson.LogBefore;
import com.bxm.shopping.common.utils.CheckUtil;
import com.bxm.shopping.dal.entity.Mold;
import com.bxm.shopping.dal.entity.Product;
import com.bxm.shopping.dal.entity.Template;
import com.bxm.shopping.model.dto.ProductDto;
import com.bxm.shopping.model.query.ProductQuery;
import com.bxm.shopping.model.vo.ProductNameAndIdVo;
import com.bxm.shopping.model.vo.ProductVo;
import com.bxm.shopping.service.IMoldService;
import com.bxm.shopping.service.IProductService;
import com.bxm.shopping.service.cache.TemplateCache;
import com.bxm.util.dto.ValidateException;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author 拉拉
 * @version v1.0.0
 * @created 2020-07-14 20:42:00
 * @modifier 拉拉
 * @updated 2020-07-14 20:42:00
 */
@Slf4j
@RestController
public class ProductController {

    @Resource
    IProductService productService;

    @Autowired
    private TemplateCache templateCache;
    @Autowired
    private IMoldService moldService;

    @PostMapping("/product/sync/syncAllProduct")
    public ResponseEntity<String> syncAllProduct(@RequestBody String data) {
        if (!data.equals("bxm-shopping")) {
            return ResponseEntity.status(500).body("验证失败");
        }
        productService.syncAllProduct();
        return ResponseEntity.ok("true");
    }

    @PostMapping("/product/sync/syncAllLp")
    public ResponseEntity<String> syncAllLp(@RequestBody String data) {
        if (!data.equals("bxm-shopping")) {
            return ResponseEntity.status(500).body("验证失败");
        }
        productService.syncAllLp();
        return ResponseEntity.ok("true");
    }

    /**
     * 新增产品
     *
     * @param request
     * @param response
     * @param productDto
     * @return
     */
    @LogBefore(operType = "/product/add", keyName = "商品新增")
    @RequestMapping(value = "/product/add", produces = "application/json", method = RequestMethod.POST)
    public ResponseEntity<String> add(HttpServletRequest request, HttpServletResponse response,
                                      @Validated @RequestBody ProductDto productDto) {

        checkParam(productDto);
        productService.addProduct(productDto);
        return ResponseEntity.ok("true");
    }

    /**
     * 修改产品
     *
     * @param request
     * @param response
     * @param productDto
     * @return
     */
    @LogBefore(operType = "/product/update", keyName = "商品编辑")
    @RequestMapping(value = "/product/update", method = RequestMethod.POST, produces = "application/json")
    public ResponseEntity<String> update(HttpServletRequest request, HttpServletResponse response,
                                         @Validated @RequestBody ProductDto productDto) {
        checkParam(productDto);
        productService.updateProduct(productDto);
        return ResponseEntity.ok("true");
    }

    /**
     * 分页查找
     *
     * @param request
     * @param response
     * @return
     */
    @RequestMapping(value = "/product/getPage", produces = "application/json")
    public ResponseEntity<Page<ProductVo>> getPage(HttpServletRequest request, HttpServletResponse response,
                                                   ProductQuery productQuery) {
        return ResponseEntity.ok(productService.getPage(productQuery));
    }


    private void checkParam(ProductDto productDto) {

        if (Objects.nonNull(productDto.getTemplateId())) {
            Template template = templateCache.getTemplateById(productDto.getTemplateId());
            if (template == null) {
                throw new ValidateException("模板不存在");
            }

            // 首部图片必传 除中间页模板
            if (!TemplateTypeEnum.NO_HEAD_IMG_TYPE.contains(template.getTemplateType()) && StringUtils.isBlank(productDto.getHeadImgs())) {
                throw new ValidateException("图片不能为空");
            }
        }


        if (productDto.getPlatformType() != null) {
            Boolean transIdCheck = productDto.getPlatformType() == 2 && StringUtils.isBlank(productDto.getTransId());
            if (transIdCheck) {
                throw new ValidateException("transId不能为空");
            }
        }
        Integer isMultiPrice = productDto.getIsMultiPrice();
        if (null != isMultiPrice && isMultiPrice == 1) {
            if (StringUtils.isBlank(productDto.getProductUnits())) {
                throw new ValidateException("productUnits不能为空");
            }
        }
        Set<String> nameSet = Sets.newHashSet();
        int nameCount = 0;
        // 多规格
        if (StringUtils.isNotBlank(productDto.getMultiProductConfig())) {
            List<ProductDto.MultiProduct> multiProducts = JSON.parseArray(productDto.getMultiProductConfig(), ProductDto.MultiProduct.class);
            for (ProductDto.MultiProduct multiProduct : multiProducts) {
                if (StringUtils.isBlank(multiProduct.getImg())) {
                    throw new ValidateException("缺少商品配置图片");
                }
                // 名称和url要么全部都有，要么全无 不可以只填一个
                boolean needNameOrUrl = (StringUtils.isBlank(multiProduct.getName()) && StringUtils.isNotBlank(multiProduct.getUrl()))
                        || (StringUtils.isNotBlank(multiProduct.getName()) && StringUtils.isBlank(multiProduct.getUrl()));
                if (needNameOrUrl) {
                    throw new ValidateException("缺少名称或url参数");
                }
                if (StringUtils.isNotBlank(multiProduct.getName())) {
                    nameCount++;
                    nameSet.add(multiProduct.getName());
                }
            }
        }
        // 摇奖配置参数
        if (StringUtils.isNotBlank(productDto.getLotteryConfig())) {
            ProductDto.LotteryConfig lotteryConfig = JSON.parseObject(productDto.getLotteryConfig(), ProductDto.LotteryConfig.class);
            conditionCheck(StringUtils.isNotBlank(lotteryConfig.getBackground()), "背景图不能为空");
            conditionCheck(StringUtils.isNotBlank(lotteryConfig.getBestAward()), "最大奖品不能为空");
            conditionCheck(StringUtils.isNotBlank(lotteryConfig.getAward1()), "奖品1不能为空");
            conditionCheck(StringUtils.isNotBlank(lotteryConfig.getAward2()), "奖品2不能为空");
        }

        if (nameSet.size() != nameCount) {
            throw new ValidateException("商品配置名称不能重复");
        }

        ProductDto.LimitStrategy limitStrategy = productDto.getLimitStrategy();

        if (null != limitStrategy) {
            conditionCheck(StringUtils.isNotBlank(limitStrategy.getLimitPhoneValue()), "limitPhoneValue 不能为空");
            conditionCheck(StringUtils.isNotBlank(limitStrategy.getLimitIdCardValue()), "limitIdCardValue 不能为空");
            conditionCheck(null != limitStrategy.getLimitAddressStatus(), "limitAddressStatus 不能为空");
            conditionCheck(null != limitStrategy.getLimitNameStatus(), "limitNameStatus 不能为空");
            conditionCheck(null != limitStrategy.getLimitPhoneStatus(), "limitPhoneStatus 不能为空");
            conditionCheck(null != limitStrategy.getLimitIdCardStatus(), "limitIdCardStatus 不能为空");
            conditionCheck(null != limitStrategy.getLimitIpBlackStatus(), "limitIpBlackStatus 不能为空");

            if (limitStrategy.getLimitPhoneValue().split("-").length != 2) {
                throw new ValidateException("limitPhoneValue参数格式错误");
            }
            if (limitStrategy.getLimitIdCardValue().split("-").length != 2) {
                throw new ValidateException("limitIdCardValue参数格式错误");
            }
        }

        if (StringUtils.isNotBlank(productDto.getLimitOrderAdvertiserId())) {
            String limitOrderAdvertiserId = productDto.getLimitOrderAdvertiserId();
            try {
                Arrays.stream(limitOrderAdvertiserId.split(",")).map(Integer::parseInt).collect(Collectors.toList());
            } catch (Exception e) {
                log.info(e.getMessage(), e);
                throw new ValidateException("订单限制多广告主id填写有误");
            }
        }


        checkProductMold(productDto);


    }

    /**
     * 互动模板配置校验
     */
    private void checkProductMold(ProductDto productDto) {
        Integer productType = productDto.getProductType();
        if (!Objects.equals(Product.PRODUCT_TYPE_INTERACTIVE, productType)) {
            return;
        }

        Integer moldId = productDto.getMoldId();

        conditionCheck(Objects.nonNull(moldId), "moldId 不能为空");
        conditionCheck(Objects.nonNull(productDto.getAdTag()), "adTag 不能为空");
        conditionCheck(StringUtils.isNotBlank(productDto.getProductMoldConfig()), "productMoldConfig不能为空");

        Mold mold = moldService.getById(moldId);
        conditionCheck(Objects.nonNull(mold), "moldId 不存在");

        if ((Objects.equals(mold.getStatus(), 0) || Objects.equals(mold.getDeleted(), 1)) && Objects.isNull(productDto.getId())) {
            throw new ValidateException("该商品模板已停用，保存失败");
        }
        if ((Objects.equals(mold.getStatus(), 0) || Objects.equals(mold.getDeleted(), 1)) && Objects.equals(productDto.getStatus(), 2)) {
            throw new ValidateException("该商品模板已停用，无法上架");
        }

        String productMoldConfig = productDto.getProductMoldConfig();
        List<Mold.ProductMoldConfig> productMoldConfigs = JSONObject.parseArray(productMoldConfig, Mold.ProductMoldConfig.class);
        if (CollectionUtils.isNotEmpty(productMoldConfigs)) {
            for (Mold.ProductMoldConfig config : productMoldConfigs) {
                conditionCheck(StringUtils.isNotBlank(config.getSceneName()), "sceneName不能为空");

                List<Mold.ProductSceneConfig> productSceneConfigs = config.getProductSceneConfigs();
                if (CollectionUtils.isNotEmpty(productSceneConfigs)) {
                    for (Mold.ProductSceneConfig productSceneConfig : productSceneConfigs) {
                        conditionCheck(StringUtils.isNotBlank(productSceneConfig.getDescription()), "description不能为空");
                    }
                }
            }
        }

    }

    /**
     * 根据id查找产品信息
     *
     * @param request
     * @param response
     * @return
     */
    @RequestMapping(value = "/product/findById", produces = "application/json")
    public ResponseEntity<ProductVo> findById(HttpServletRequest request, HttpServletResponse response,
                                              @RequestParam(value = "id") Integer id) {
        return ResponseEntity.ok(productService.findById(id));
    }

    /**
     * 根据id修改状态
     *
     * @param request
     * @param response
     * @param id
     * @param status
     * @return
     */
    @RequestMapping(value = "/product/updateStatus", method = RequestMethod.POST, produces = "application/json")
    public ResponseEntity<String> updateStatus(HttpServletRequest request, HttpServletResponse response,
                                               @RequestParam(value = "id") Integer id,
                                               @RequestParam(value = "status") Integer status) {
        productService.updateStatus(id, status);
        return ResponseEntity.ok("true");
    }


    /**
     * 根据id修改防封域名组
     *
     * @param id
     * @param groupId
     * @return
     */
    @GetMapping(value = "/product/updateLpKylinGroupId")
    public ResponseEntity<String> updateLpKylinGroupId(@RequestParam Integer id, @RequestParam(required = false) String groupId) {
        productService.updateLpKylinGroupId(id, groupId);
        return ResponseEntity.ok("true");
    }


    /**
     * 根据id修改状态
     *
     * @param request
     * @param response
     * @param ids
     * @param status
     * @return
     */
    @RequestMapping(value = "/product/batchUpdateStatus", method = RequestMethod.POST, produces = "application/json")
    public ResponseEntity<String> batchUpdateStatus(HttpServletRequest request, HttpServletResponse response,
                                                    @RequestParam(value = "ids") String ids,
                                                    @RequestParam(value = "status") Integer status) {
        productService.batchUpdateStatus(ids, status);
        return ResponseEntity.ok("true");
    }


    /**
     * 查找所有
     *
     * @param request
     * @param response
     * @return
     */
    @RequestMapping(value = "/product/findAll", produces = "application/json")
    public ResponseEntity<List<ProductNameAndIdVo>> findAll(HttpServletRequest request, HttpServletResponse response) {
        return ResponseEntity.ok(productService.findAll());
    }


    /**
     * 条件检查
     *
     * @param expression
     * @param errorMessage
     */
    public static void conditionCheck(boolean expression, String errorMessage) {
        CheckUtil.conditionCheck(expression, errorMessage);
    }
}
