package com.bxm.spider.deal.dal.service.impl;

import com.baomidou.mybatisplus.service.impl.ServiceImpl;
import com.bxm.spider.cache.RedisClient;
import com.bxm.spider.cache.constant.IdKeyConstant;
import com.bxm.spider.constant.exception.BaseRunException;
import com.bxm.spider.constant.monitor.MonitorConstant;
import com.bxm.spider.constant.monitor.MonitorHelper;
import com.bxm.spider.constant.processor.ProcessorParameter;
import com.bxm.spider.deal.common.constant.ContentConstant;
import com.bxm.spider.deal.common.utils.StringHelp;
import com.bxm.spider.deal.dal.mapper.NewsMapper;
import com.bxm.spider.deal.dal.service.NewsContentService;
import com.bxm.spider.deal.dal.service.NewsService;
import com.bxm.spider.deal.dal.utils.MultiPageHelps;
import com.bxm.spider.deal.facade.model.ContentDto;
import com.bxm.spider.deal.facade.model.ContentVo;
import com.bxm.spider.deal.model.dao.News;
import com.bxm.spider.deal.model.dao.NewsContent;
import com.bxm.spider.utils.DateUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * <p>
 * 爬取网页内容表(分表) 服务实现类
 * </p>
 *
 * @author JandMin
 * @since 2018-10-16
 */
@Service
public class NewsServiceImpl extends ServiceImpl<NewsMapper, News> implements NewsService {
    private static final Logger LOG = LoggerFactory.getLogger(NewsServiceImpl.class);
    @Autowired
    private NewsMapper newsMapper;
    @Autowired
    private NewsContentService newsContentService;
    @Autowired
    private RedisClient redisClient;

    @Override
    public News findNewsByUrl(String url) {
        return newsMapper.findNewsByUrl(StringHelp.getHashCodeModTen(url), url.hashCode());
    }

    @Override
    public int insertNews(News news) {
        news.setSuffix(StringHelp.getHashCodeModTen(news.getUrl()));
        Long id = redisClient.hincrByOne(IdKeyConstant.getPrimaryId(), IdKeyConstant.FIELD_NEWS);
        if (null == id) {
            id = StringHelp.getUUIDHashCode(14);
        }
        news.setId(id * 10 + Long.parseLong(news.getSuffix()));
        return newsMapper.insertReturnId(news);
    }

    @Override
    public int updateNewsById(News news) {
        if (StringUtils.isBlank(news.getSuffix())) {
            news.setSuffix(StringHelp.getHashCodeModTen(news.getUrl()));
        }
        return newsMapper.updateNewsById(news);
    }

    @Transactional(rollbackFor = {Exception.class, BaseRunException.class}, timeout = 120)
    @Override
    public long insertNewsAndContent(News news, NewsContent newsContent, ProcessorParameter processorParameter) {
        // 参数校验
        if (StringUtils.isBlank(newsContent.getContent())) {
            throw new BaseRunException("[save data]news content is null,the url is " + news.getUrl());
        }
        long result = 0;
        // 判断url是否存在
        News newsOld = findNewsByUrl(news.getUrl());
        news.setUrlCode(news.getUrl().hashCode());
        news.setModifyTime(new Date());
        if (null == newsOld) {
            //保存新闻数据
            insertNews(news);
            //保存新闻内容
            if (null == news.getId()) {
                throw new BaseRunException("the news id is null when save news,so can't save news content the url is " + news.getUrl());
            }
            result = news.getId();
            newsContent.setId(result);
            newsContentService.insert(news.getUrl(), newsContent);

            LOG.info(MonitorConstant.MONITOR, "新闻数据插入成功 {}", MonitorHelper.ofSuccessLog(MonitorConstant.SAVE_INSERT, processorParameter));
        } else {
            result = newsOld.getId();
            news.setId(result);
            if (ContentConstant.STATUS_RENEWAL == newsOld.getStatus().intValue()) {
                news.setStatus(ContentConstant.STATUS_VALID);
            } else {
                news.setStatus(newsOld.getStatus());
            }
            updateNewsById(news);
            newsContent.setId(result);
            newsContentService.updateById(news.getUrl(), newsContent);

            LOG.info(MonitorConstant.MONITOR, "新闻数据更新成功 {}", MonitorHelper.ofSuccessLog(MonitorConstant.SAVE_UPDATE, processorParameter));
        }
        return result;
    }

    @Override
    public List<Map<String, String>> findNewsByModifyTime(int amount) {
        Date modifyTime = DateUtils.addField(new Date(), Calendar.DAY_OF_YEAR, amount);
        List<Map<String, String>> newsList = newsMapper.findNewsByModifyTime(modifyTime);
        return newsList;
    }

    @Override
    public Map<String, String> findNewsById(Long id, String suffix) {
        return newsMapper.findNewsById(id, suffix);
    }

    @Override
    public List<ContentVo> listContent(ContentDto contentDto) {
        return MultiPageHelps.list(contentDto, newsMapper);
    }

}
