package com.bxm.warcar.integration.autoconfigure.distributed;

import com.bxm.warcar.integration.distributed.DistributedLock;
import com.bxm.warcar.integration.distributed.DistributedLockBus;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.context.annotation.Configuration;

import java.lang.reflect.Method;

/**
 * @author allen
 * @since 1.0.0
 */
@Aspect
@Configuration
@AutoConfigureAfter(DistributedScheduledAutoConfiguration.class)
@ConditionalOnBean(DistributedLockBus.class)
public class DistributedScheduled {

    private final static Logger LOGGER = LoggerFactory.getLogger(DistributedScheduled.class);

    private final DistributedLockBus distributedLockBus;

    public DistributedScheduled(DistributedLockBus distributedLockBus) {
        this.distributedLockBus = distributedLockBus;
    }

    @Pointcut("@annotation(org.springframework.scheduling.annotation.Scheduled)")
    public void pointcut() {
    }

    @Around("pointcut()")
    public Object around(ProceedingJoinPoint point) throws Throwable {
        boolean acquire = false;
        DistributedLock lock = null;
        try {
            Object target = point.getTarget();
            Method method = getMethod(point);
            if (null == method) {
                return point.proceed();
            }
            lock = distributedLockBus.getLock(target, method);
            if (null == lock) {
                return point.proceed();
            }
            acquire = lock.acquire();
            if (!acquire) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Distributed job {}#{} already execution at other machine.",
                            target.getClass(), method.getName());
                }
                return null;
            }
            Object proceed = point.proceed();
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("Distributed job {}#{} has been executed.", target.getClass(), method.getName());
            }
            return proceed;
        } finally {
            if (null != lock && acquire) {
                lock.release();
            }
        }
    }

    private Method getMethod(ProceedingJoinPoint point) {
        MethodSignature methodSignature = (MethodSignature) point.getSignature();

        Class<?> targetClass = point.getTarget().getClass();
        try {
            return targetClass.getMethod(methodSignature.getName(), methodSignature.getParameterTypes());
        } catch (NoSuchMethodException e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("getMethod:", e);
            }
            return null;
        }
    }
}
