package com.bxm.warcar.logging.autoconfigure;

import com.bxm.warcar.logging.LoggingWriter;
import com.bxm.warcar.logging.OncePerRequestLoggingFilter;
import com.bxm.warcar.logging.OperatorExtractor;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.core.Ordered;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.util.ClassUtils;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.function.Supplier;

/**
 * @author allen
 * @since 2020-03-18
 */
@EnableConfigurationProperties(LoggingProperties.class)
@ConditionalOnProperty(prefix = "warcar.logging", name = "enable", havingValue = "true")
public class LoggingAutoConfiguration {

    private static final Logger LOGGER = LoggerFactory.getLogger(LoggingAutoConfiguration.class);
    private final LoggingProperties loggingProperties;

    public LoggingAutoConfiguration(LoggingProperties loggingProperties) {
        this.loggingProperties = loggingProperties;
    }

    @Bean
    public OncePerRequestLoggingFilter oncePerRequestLoggingFilter(
            ConfigurableEnvironment environment,
            ObjectProvider<List<LoggingWriter>> loggingWriterObjectProvider,
            ObjectProvider<OperatorExtractor> operatorExtractorObjectProvider) throws Throwable {
        String applicationId = getApplicationId(environment);
        List<LoggingWriter> writers = loggingWriterObjectProvider.getIfAvailable();

        LoggingWriter loggingWriter = null;
        Class<? extends LoggingWriter> writer = loggingProperties.getWriter();
        if (null != writer && CollectionUtils.isNotEmpty(writers)) {
            loggingWriter = writers.stream().filter(o -> (ClassUtils.getUserClass(o).equals(writer))).findAny()
                    .orElseThrow((Supplier<Throwable>) () -> new NoSuchBeanDefinitionException(writer));
            LOGGER.info("Using {}", loggingWriter);
        }

        OperatorExtractor operatorExtractor = operatorExtractorObjectProvider.getIfAvailable();
        if (null == operatorExtractor) {
            LOGGER.warn("No OperatorExtractor implementor!");
        }

        int payloadByteMaxLength = loggingProperties.getPayloadByteMaxLength();
        return new OncePerRequestLoggingFilter(applicationId, payloadByteMaxLength, loggingWriter, operatorExtractor);
    }

    @Bean
    public FilterRegistrationBean<OncePerRequestLoggingFilter> oncePerRequestLoggingFilterFilterRegistrationBean(OncePerRequestLoggingFilter oncePerRequestLoggingFilter) {
        FilterRegistrationBean<OncePerRequestLoggingFilter> bean = new FilterRegistrationBean<>(oncePerRequestLoggingFilter);
        bean.setDispatcherTypes(loggingProperties.getDispatcherType());
        bean.addUrlPatterns(loggingProperties.getUrlPatterns());
        bean.setName(loggingProperties.getFilterName());
        bean.setOrder(Ordered.LOWEST_PRECEDENCE);
        return bean;
    }

    private String getApplicationId(ConfigurableEnvironment environment) {
        String name = environment.getProperty("spring.application.name");
        return StringUtils.hasText(name) ? name : "application";
    }
}
