package com.bxm.warcar.cache;

import java.util.Map;
import java.util.function.Predicate;

/**
 * 计数器
 *
 * @author allen
 * @date 2017-12-04
 */
public interface Counter {

    /**
     * 原子操作，对key进行加1操作，并返回操作后的结果
     *
     * @param keyGenerator 键生成器
     * @return 操作后的结果
     */
    Long incrementAndGet(KeyGenerator keyGenerator);

    /**
     * 原子操作，对key进行加1操作，并返回操作后的结果
     *
     * @param keyGenerator       键生成器
     * @param expireTimeInSecond 过期时间（秒）
     * @return 操作后的结果
     */
    Long incrementAndGet(KeyGenerator keyGenerator, int expireTimeInSecond);

    /**
     * 原子操作，对key进行加1操作，并返回操作后的结果
     *
     * @param keyGenerator       键生成器
     * @param expireTimeInSecond 过期时间（秒）
     * @param updateExpireTime   当过期时间更新策略返回true时，才会更新过期时间，入参为操作后的结果
     * @return 操作后的结果
     */
    Long incrementAndGet(KeyGenerator keyGenerator, int expireTimeInSecond, Predicate<Long> updateExpireTime);

    /**
     * 原子操作，对key进行加inc操作，并返回操作后的结果
     *
     * @param keyGenerator 键生成器
     * @param inc          增量值
     * @return 操作后的结果
     */
    Long incrementByAndGet(KeyGenerator keyGenerator, long inc);

    /**
     * 原子操作，对key进行加inc操作，并返回操作后的结果
     *
     * @param keyGenerator       键生成器
     * @param inc                增量值
     * @param expireTimeInSecond 过期时间（秒）
     * @return 操作后的结果
     */
    Long incrementByAndGet(KeyGenerator keyGenerator, long inc, int expireTimeInSecond);

    /**
     * 原子操作，对key进行加inc操作，并返回操作后的结果
     *
     * @param keyGenerator       键生成器
     * @param inc                增量值
     * @param expireTimeInSecond 过期时间（秒）
     * @param updateExpireTime   当过期时间更新策略返回true时，才会更新过期时间，入参为操作后的结果
     * @return 操作后的结果
     */
    Long incrementByAndGet(KeyGenerator keyGenerator, long inc, int expireTimeInSecond, Predicate<Long> updateExpireTime);

    /**
     * 原子操作，对key的浮点值进行增加操作
     *
     * @param keyGenerator 键生成器
     * @param inc          增量值
     * @return 操作后的结果
     */
    Double incrFloatByAndGet(KeyGenerator keyGenerator, double inc);

    /**
     * 原子操作，对key的浮点值进行增加操作
     *
     * @param keyGenerator       键生成器
     * @param inc                增量值
     * @param expireTimeInSecond 过期时间（秒）
     * @return 操作后的结果
     */
    Double incrFloatByAndGet(KeyGenerator keyGenerator, double inc, int expireTimeInSecond);

    /**
     * 原子操作，对key进行减1操作，并返回操作后的结果
     *
     * @param keyGenerator 键生成器
     * @return 操作后的结果
     */
    Long decrementAndGet(KeyGenerator keyGenerator);

    /**
     * 原子操作，对key进行减1操作，并返回操作后的结果
     *
     * @param keyGenerator       键生成器
     * @param expireTimeInSecond 过期时间（秒）
     * @return 操作后的结果
     */
    Long decrementAndGet(KeyGenerator keyGenerator, int expireTimeInSecond);

    /**
     * 原子操作，对key进行减dec操作，并返回操作后的结果
     *
     * @param keyGenerator 键生成器
     * @param dec          减量值
     * @return 操作后的结果
     */
    Long decrementByAndGet(KeyGenerator keyGenerator, long dec);

    /**
     * 原子操作，对key进行减dec操作，并返回操作后的结果
     *
     * @param keyGenerator       键生成器
     * @param dec                减量值
     * @param expireTimeInSecond 过期时间（秒）
     * @return 操作后的结果
     */
    Long decrementByAndGet(KeyGenerator keyGenerator, long dec, int expireTimeInSecond);

    /**
     * 设置key的计数值
     *
     * @param keyGenerator 键生成器
     * @param value        计数值
     */
    void set(KeyGenerator keyGenerator, long value);

    /**
     * 重新设置key的过期时间
     *
     * @param keyGenerator 键生成器
     * @param seconds      过期时间（秒）
     */
    void expire(KeyGenerator keyGenerator, int seconds);

    /**
     * 获取key的当前计数值
     *
     * @param keyGenerator 键生成器
     * @return 当前计数值, 如果key不存在则返回null
     */
    Long get(KeyGenerator keyGenerator);

    /**
     * 原子操作，对哈希表中的field进行加1操作
     *
     * @param keyGenerator 键生成器
     * @param field        哈希表中的字段
     * @return 操作后的结果
     */
    Long hincrementAndGet(KeyGenerator keyGenerator, String field);

    /**
     * 原子操作，对哈希表中的field进行加1操作
     *
     * @param keyGenerator       键生成器
     * @param field              哈希表中的字段
     * @param expireTimeInSecond 过期时间（秒）
     * @return 操作后的结果
     */
    Long hincrementAndGet(KeyGenerator keyGenerator, String field, int expireTimeInSecond);

    /**
     * 原子操作，对哈希表中的field进行加1操作
     *
     * @param keyGenerator       键生成器
     * @param field              哈希表中的字段
     * @param expireTimeInSecond 过期时间（秒）
     * @param updateExpireTime   当过期时间更新策略返回true时，才会更新过期时间，入参为操作后的结果
     * @return 操作后的结果
     */
    Long hincrementAndGet(KeyGenerator keyGenerator, String field, int expireTimeInSecond, Predicate<Long> updateExpireTime);

    /**
     * 原子操作，对哈希表中的field进行加inc操作
     *
     * @param keyGenerator 键生成器
     * @param field        哈希表中的字段
     * @param inc          增量值
     * @return 操作后的结果
     */
    Long hincrementByAndGet(KeyGenerator keyGenerator, String field, long inc);

    /**
     * 原子操作，对哈希表中的field进行加inc操作
     *
     * @param keyGenerator       键生成器
     * @param field              哈希表中的字段
     * @param inc                增量值
     * @param expireTimeInSecond 过期时间（秒）
     * @return 操作后的结果
     */
    Long hincrementByAndGet(KeyGenerator keyGenerator, String field, long inc, int expireTimeInSecond);

    /**
     * 原子操作，对哈希表中的field进行加inc操作
     *
     * @param keyGenerator       键生成器
     * @param field              哈希表中的字段
     * @param inc                增量值
     * @param expireTimeInSecond 过期时间（秒）
     * @param updateExpireTime   当过期时间更新策略返回true时，才会更新过期时间，入参为操作后的结果
     * @return 操作后的结果
     */
    Long hincrementByAndGet(KeyGenerator keyGenerator, String field, long inc, int expireTimeInSecond, Predicate<Long> updateExpireTime);

    /**
     * 原子操作，对哈希表中的field的浮点值进行增加操作
     *
     * @param keyGenerator 键生成器
     * @param field        哈希表中的字段
     * @param inc          增量值
     * @return 操作后的结果
     */
    Double hincrFloatByAndGet(KeyGenerator keyGenerator, String field, double inc);

    /**
     * 原子操作，对哈希表中的field的浮点值进行增加操作
     *
     * @param keyGenerator       键生成器
     * @param field              哈希表中的字段
     * @param inc                增量值
     * @param expireTimeInSecond 过期时间（秒）
     * @return 操作后的结果
     */
    Double hincrFloatByAndGet(KeyGenerator keyGenerator, String field, double inc, int expireTimeInSecond);

    /**
     * 获取哈希表中指定field的值
     *
     * @param keyGenerator 键生成器
     * @param field        哈希表中的字段
     * @return field对应的值, 如果field不存在则返回0
     */
    Long hget(KeyGenerator keyGenerator, String field);

    /**
     * 获取哈希表中所有的field和value
     *
     * @param keyGenerator 键生成器
     * @return 包含所有field和value的Map
     */
    Map<String, Long> hgetall(KeyGenerator keyGenerator);

    /**
     * 返回这个计数器具体的实现原生客户端
     *
     * @return 原生客户端实例
     */
    Object getClientOriginal();
}
