package com.bxm.warcar.canal.mq;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.otter.canal.protocol.CanalEntry;
import com.bxm.warcar.canal.CanalClient;
import com.bxm.warcar.canal.CanalEntity;
import com.bxm.warcar.canal.CanalEventListener;
import com.bxm.warcar.canal.CanalEventType;
import com.bxm.warcar.mq.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

/**
 * @author allen
 * @since 1.0.0
 */
public class CanalMessageQueueClient extends CanalClient {

    private final static Logger LOGGER = LoggerFactory.getLogger(CanalMessageQueueClient.class);

    private final Producer producer;
    private final Consumer consumer;
    private final String topic;

    public CanalMessageQueueClient(Producer producer, Consumer consumer, String topic, String zkServers, String destination) {
        this(producer, consumer, topic, zkServers, destination, null, null);
    }

    public CanalMessageQueueClient(Producer producer, Consumer consumer, String topic, String zkServers, String destination, String username, String password) {
        super(zkServers, destination, username, password);
        this.producer = producer;
        this.consumer = consumer;
        this.topic = topic;

        if (!producer.isStarted()) {
            producer.start();
        }
        if (!consumer.isStarted()) {
            consumer.start();
        }

        Listener messageListener = consumer.getMessageListener();
        if (!(messageListener instanceof CanalMessageListenerDispatcher)) {
            throw new RuntimeException("This message listener must be instance CanalMessageListenerDispatcher for the consumer.");
        }
        super.addListeners(((CanalMessageListenerDispatcher) messageListener).getListeners());
    }

    @Override
    public void addListener(CanalEventListener<? extends CanalEntity> listener) {
        throw new UnsupportedOperationException("Please use com.bxm.warcar.canal.mq.CanalMessageListenerDispatcher#(String, CanalEventListener<? extends CanalEntity>)");
    }

    @Override
    protected <T extends CanalEntity> void doDelete(List<CanalEntry.Column> columns, CanalEventListener<T> listener) {
        T object = convert(columns, listener);
        if (null == object) {
            return;
        }
        CanalMessageBody entity = new CanalMessageBody(CanalEventType.DELETE, object, null);
        sendMessage(listener, entity);
    }

    @Override
    protected <T extends CanalEntity> void doInsert(List<CanalEntry.Column> columns, CanalEventListener<T> listener) {
        T object = convert(columns, listener);
        if (null == object) {
            return;
        }
        CanalMessageBody entity = new CanalMessageBody(CanalEventType.INSERT, object, null);
        sendMessage(listener, entity);
    }

    @Override
    protected <T extends CanalEntity> void doUpdate(List<CanalEntry.Column> before, List<CanalEntry.Column> after, CanalEventListener<T> listener) {
        T beforeObj = convert(before, listener);
        if (null == beforeObj) {
            return;
        }
        T afterObj = convert(after, listener);
        if (null == afterObj) {
            return;
        }
        CanalMessageBody entity = new CanalMessageBody(CanalEventType.UPDATE, beforeObj, afterObj);
        sendMessage(listener, entity);
    }

    private <T extends CanalEntity> void sendMessage(CanalEventListener<T> listener, CanalMessageBody entity) {
        try {
            Message message = new Message();
            message.setTopic(topic);
            message.setTags(listener.listening());
            message.setBody(JSONObject.toJSONBytes(entity));
            SendResult result = producer.send(message);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("message [{}] send successful", result.getMsgId());
            }
        } catch (SendException e) {
            // try again?
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("send:", e);
            }
            throw new RuntimeException("send:", e);
        }
    }
}
