package com.bxm.warcar.datasync.server.sync.zk;

import com.bxm.warcar.datasync.server.sync.SyncCache;
import com.bxm.warcar.datasync.server.sync.SyncController;
import com.bxm.warcar.zk.ZkClientHolder;
import com.google.common.base.Preconditions;
import org.apache.commons.lang.StringUtils;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.utils.ZKPaths;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author allen
 * @date 2019/4/2
 * @since 1.0.0
 */
public class ZkSyncController implements SyncController {

    private static final Logger LOGGER = LoggerFactory.getLogger(ZkSyncController.class);
    public static final String DEFAULT_PREFIX_PATH = "/warcar/data_sync";

    private final ZkClientHolder zkClientHolder;
    private final String prefixPath;

    public ZkSyncController(ZkClientHolder zkClientHolder) {
        this(zkClientHolder, DEFAULT_PREFIX_PATH);
    }

    public ZkSyncController(ZkClientHolder zkClientHolder, String prefixPath) {
        Preconditions.checkNotNull(zkClientHolder);
        Preconditions.checkArgument(StringUtils.isNotBlank(prefixPath));
        this.zkClientHolder = zkClientHolder;
        this.prefixPath = prefixPath;
    }

    @Override
    public boolean update(String key, Class<?> cls) {
        CuratorFramework framework = zkClientHolder.get();

        Preconditions.checkNotNull(framework, "framework");
        Preconditions.checkArgument(StringUtils.isNotBlank(key), "key");

        String path = ZKPaths.makePath(prefixPath, key);

        try {
            Stat stat = framework.checkExists().forPath(path);
            SyncCache syncCache = new SyncCache(key, cls);

            byte[] data = syncCache.toByte();
            if (null == stat) {
                framework.create().creatingParentsIfNeeded().forPath(path, data);
            } else {
                framework.setData().forPath(path, data);
            }
            return true;
        } catch (Exception e) {
            LOGGER.error("zk operation: " + path, e);
        }

        return false;
    }

    @Override
    public void delete(String key) {
        CuratorFramework framework = zkClientHolder.get();

        Preconditions.checkNotNull(framework, "framework");
        Preconditions.checkArgument(StringUtils.isNotBlank(key), "key");
        String path = ZKPaths.makePath(prefixPath, key);
        try {
            framework.delete().guaranteed().forPath(path);
        } catch (Exception e) {
            LOGGER.error("zk operation: " + path, e);
        }
    }
}
