/*
 * Copyright 2016 bianxianmao.com All right reserved. This software is the confidential and proprietary information of
 * textile.com ("Confidential Information"). You shall not disclose such Confidential Information and shall use it only
 * in accordance with the terms of the license agreement you entered into with bianxianmao.com.
 */

package com.bxm.warcar.dpl.impl;

import com.bxm.warcar.dpl.Plugin;
import com.bxm.warcar.dpl.PluginConfig;
import org.apache.commons.logging.LogFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.CachedIntrospectionResults;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.handler.AbstractHandlerMethodMapping;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;

import java.beans.Introspector;
import java.util.Date;
import java.util.Map;
import java.util.ResourceBundle;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * @author allen
 * @since 1.0.0
 */
public class SpringPlugin implements Plugin {

    private static final Logger LOGGER = LoggerFactory.getLogger(SpringPlugin.class);

    private final PluginConfig pluginConfig;
    private final Date creation;
    private final PluginApplicationContext applicationContext;

    public SpringPlugin(PluginConfig pluginConfig, PluginApplicationContext applicationContext) {
        this.pluginConfig = pluginConfig;
        this.creation = new Date();
        this.applicationContext = applicationContext;
    }

    @Override
    public String getName() {
        return pluginConfig.getName();
    }

    @Override
    public String getVersion() {
        return pluginConfig.getVersion();
    }

    @Override
    public Date getCreation() {
        return creation;
    }

    @Override
    public void destroy() {
        unregisterMapping();
        closeQuietly(applicationContext);
        clear(applicationContext.getClassLoader());
    }

    private void unregisterMapping() {
        Map<RequestMappingInfo, HandlerMethod> mappings = applicationContext.getMappings();
        AbstractHandlerMethodMapping handlerMethodMapping = applicationContext.getBean(AbstractHandlerMethodMapping.class);
        for (RequestMappingInfo requestMappingInfo : mappings.keySet()) {
            handlerMethodMapping.unregisterMapping(requestMappingInfo);
        }
    }

    /**
     * 清除类加载器
     *
     * @param classLoader
     */
    public static void clear(ClassLoader classLoader) {
        checkNotNull(classLoader, "classLoader is null");
        //Introspector缓存BeanInfo类来获得更好的性能。卸载时刷新所有Introspector的内部缓存。
        Introspector.flushCaches();
        //从已经使用给定类加载器加载的缓存中移除所有资源包
        ResourceBundle.clearCache(classLoader);
        //Clear the introspection cache for the given ClassLoader
        CachedIntrospectionResults.clearClassLoader(classLoader);
//        LogFactory.release(classLoader);
    }

    /**
     * 关闭Spring上下文
     * @param applicationContext
     */
    private static void closeQuietly(ConfigurableApplicationContext applicationContext) {
        checkNotNull(applicationContext, "applicationContext is null");
        try {
            applicationContext.close();
        } catch (Exception e) {
            LOGGER.error("Failed to close application context", e);
        }
    }

    @Override
    public ClassLoader getChildClassLoader() {
        return this.applicationContext.getClassLoader();
    }

    @Override
    public PluginConfig getPluginConfig() {
        return pluginConfig;
    }
}
