package com.bxm.warcar.dpl.hotswap;

import com.bxm.warcar.dpl.Plugin;
import com.bxm.warcar.dpl.PluginConfig;
import com.bxm.warcar.dpl.PluginLoader;
import com.bxm.warcar.dpl.PluginManager;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.MapDifference;
import com.google.common.collect.Maps;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang3.concurrent.BasicThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import java.util.*;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import static com.google.common.base.Preconditions.checkNotNull;


public abstract class AbstractPluginRefreshScheduler implements InitializingBean, DisposableBean, Runnable {

    private static final Logger LOGGER = LoggerFactory
            .getLogger(AbstractPluginRefreshScheduler.class);

    /**
     * 默认延迟执行,单位秒
     */
    private static final int DEFAULT_INITIAL_DELAY = 5;

    /**
     * 插件刷新默认间隔,单位秒
     */
    private static final int DEFAULT_REFRESH_DELAY = 1;

    /** 初始化的延迟时间 */
    private int initialDelay = DEFAULT_INITIAL_DELAY;

    /** 刷新间隔时间 */
    private int refreshDelay = DEFAULT_REFRESH_DELAY;

    private ScheduledExecutorService scheduledExecutor;
    private final PluginManager pluginManager;
    private final PluginLoader pluginLoader;

    public AbstractPluginRefreshScheduler(PluginManager pluginManager, PluginLoader pluginLoader) {
        this.pluginManager = pluginManager;
        this.pluginLoader = pluginLoader;
    }

    /**
     * 初始化ScheduledExecutor，启动定时任务，扫描数据库的PluginConfig，并根据逻辑判断启动和卸载插件
     *
     * @see InitializingBean#afterPropertiesSet()
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        //先刷新一次
        refreshPluginConfigs();
        scheduledExecutor = new ScheduledThreadPoolExecutor(1,
                new BasicThreadFactory.Builder().namingPattern("plugin_refresh-schedule-pool-%d").daemon(true).build());
        scheduledExecutor.scheduleWithFixedDelay(this, initialDelay, refreshDelay, TimeUnit.SECONDS);
        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("AbstractPluginRefreshScheduler start");
        }
    }

    /**
     * 关闭ScheduledExecutor
     * @see DisposableBean#destroy()
     */
    @Override
    public void destroy() throws Exception {
        if (scheduledExecutor != null) {
            scheduledExecutor.shutdownNow();
        }
    }

    /**
     * ScheduledExecutor 定时运行的方法
     * @see Runnable#run()
     */
    @Override
    public void run() {
        try {
            refreshPluginConfigs();
        } catch (Throwable e) {
            LOGGER.error("Failed to refresh plugin configs", e);
        }
    }

    /**
     * 获取插件配置信息
     *
     * @return
     */
    public abstract List<PluginConfig> queryPluginConfigs();

    /**
     * 刷新PluginConfig
     */
    private void refreshPluginConfigs() {
        // 查找状态为ENABLED的PluginConfig，并以插件名作为Key，放到Map中
        Map<String, PluginConfig> moduleConfigs = indexPluginConfigByModuleName(filterEnabledModule());

        // 转换Map的Value，提取Plugin的Version，Map的Key为DataProvider，Value为Version
        Map<String, String> configVersions = transformToConfigVersions(moduleConfigs);
        // 获取当前内存中，也就是PluginManager已经加载的模板版本，同样Map的Key为name，Value为Version
        Map<String, String> moduleVersions = transformToModuleVersions(pluginManager.getPlugins());
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Config size: {}", configVersions.size());
            LOGGER.debug("Plugin size: {}", moduleVersions.size());
            LOGGER.debug("now in map {}", moduleVersions);
        }
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Config versions: {}", configVersions);
            LOGGER.debug("Plugin versions: {}", moduleVersions);
        }
        // 找出配置与当前内存里配置的不同
        MapDifference<String, String> difference = Maps.difference(configVersions, moduleVersions);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Version difference: {}", difference);
        }
        // 配置新增的
        putPlugins(moduleConfigs, configAdds(difference));
        // 配置版本与插件不同的
        putPlugins(moduleConfigs, configDifference(difference));
        // 插件多余的
        removeModules(pluginsRedundant(difference));
    }

    /**
     * 查找状态为ENABLED的PluginConfig
     */
    private Collection<PluginConfig> filterEnabledModule() {
        List<PluginConfig> pluginConfigs = queryPluginConfigs();
        if (pluginConfigs == null || pluginConfigs.isEmpty()) {
            return new ArrayList<PluginConfig>();
        }
        return Collections2.filter(pluginConfigs, new Predicate<PluginConfig>() {
            @Override
            public boolean apply(PluginConfig moduleConfig) {
                return moduleConfig.isEnabled();
            }
        });
    }

    /**
     * 根据dataProviders指定的PluginConfig初始化插件，并放入PluginManager中
     *
     * @param pluginConfigs
     * @param pluginNames
     */
    private void putPlugins(Map<String, PluginConfig> pluginConfigs, Set<String> pluginNames) {
        for (String name : pluginNames) {
            PluginConfig pluginConfig = pluginConfigs.get(name);
            try {
                if (isFailedVersion(pluginConfig)) {
                    if (LOGGER.isInfoEnabled()) {
                        LOGGER.info("this version is failed, ignore.{}", pluginConfig);
                    }
                    continue;
                }
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Load plugin config: {}", pluginConfig);
                }
                Plugin plugin = pluginLoader.load(pluginConfig);
                Plugin removed = pluginManager.register(plugin);
                destroyQuietly(removed);
                pluginManager.getErrorPluginContext().remove(name.toUpperCase(Locale.CHINESE));
                pluginManager.getErrorPluginContext().remove(name.toUpperCase(Locale.CHINESE) + "_ERROR");
            } catch (Exception e) {
                pluginManager.getErrorPluginContext().put(name.toUpperCase(Locale.CHINESE) + "_ERROR",
                        ToStringBuilder.reflectionToString(e));
                pluginManager.getErrorPluginContext().put(name.toUpperCase(Locale.CHINESE),
                        pluginConfig.getVersion());
                LOGGER.error("Failed to load plugin config: " + pluginConfig, e);
            } catch (Error e) {
                LOGGER.error("Failed to load plugin config: " + pluginConfig, e);
            }
        }
    }

    /**
     *
     * @param pluginConfig
     * @return
     */
    private boolean isFailedVersion(PluginConfig pluginConfig) {
        checkNotNull(pluginConfig, "pluginConfig is null");
        String name = pluginConfig.getName();
        checkNotNull(name, "name is null");
        String version = pluginManager.getErrorPluginContext().get(name.toUpperCase(Locale.CHINESE));
        return pluginConfig.getVersion().equals(version);
    }

    /**
     * 移除并且卸载插件
     *
     * @param pluginsRedundant
     */
    private void removeModules(Set<String> pluginsRedundant) {
        for (String name : pluginsRedundant) {
            Plugin removed = pluginManager.remove(name);
            destroyQuietly(removed);
        }
    }

    /**
     * 销毁插件，不抛出异常
     *
     * @param module
     */
    private static void destroyQuietly(Plugin module) {
        if (module != null) {
            try {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Destroy plugin: {}", module.getName());
                }
                module.destroy();
            } catch (Exception e) {
                LOGGER.error("Failed to destroy plugin " + module, e);
            }
        }
    }

    /**
     * 根据对比的结果，查找多余的插件，
     *
     * @param difference
     * @return
     */
    private Set<String> pluginsRedundant(MapDifference<String, String> difference) {
        return difference.entriesOnlyOnRight().keySet();
    }

    /**
     * 根据对比结果，查找版本不同的插件
     *
     * @param difference
     * @return
     */
    private Set<String> configDifference(MapDifference<String, String> difference) {
        return difference.entriesDiffering().keySet();
    }

    /**
     * 根据对比结果，查找新增的插件
     *
     * @param difference
     * @return
     */
    private Set<String> configAdds(MapDifference<String, String> difference) {
        return difference.entriesOnlyOnLeft().keySet();
    }

    /**
     * 将一个Plugin List，转换为Map，Key为 name，Value为Version
     *
     * @param modules
     * @return
     */
    private Map<String, String> transformToModuleVersions(List<Plugin> modules) {
        return ImmutableMap.copyOf(Maps.transformValues(
                Maps.uniqueIndex(modules, new Function<Plugin, String>() {
                    @Override
                    public String apply(Plugin input) {
                        return input.getName();
                    }
                }), new Function<Plugin, String>() {
                    @Override
                    public String apply(Plugin input) {
                        return input.getVersion();
                    }
                }));
    }

    /**
     * 提取Map中Value的Version，转换成新的Map，Key为name，Value为Version
     *
     * @param moduleConfigs
     * @return
     */
    private Map<String, String> transformToConfigVersions(Map<String, PluginConfig> moduleConfigs) {
        return ImmutableMap.copyOf(Maps.transformValues(moduleConfigs,
                new Function<PluginConfig, String>() {
                    @Override
                    public String apply(PluginConfig input) {
                        return input.getVersion();
                    }
                }));
    }

    /**
     * 将PluginConfig List转换成为Map，Key为name，Value为PluginConfig
     *
     * @param list
     * @return
     */
    private Map<String, PluginConfig> indexPluginConfigByModuleName(Collection<PluginConfig> list) {

        return ImmutableMap.copyOf(Maps.uniqueIndex(list, new Function<PluginConfig, String>() {
            @Override
            public String apply(PluginConfig input) {
                return input.getName();
            }
        }));
    }

    public void setInitialDelay(int initialDelay) {
        this.initialDelay = initialDelay;
    }

    public void setRefreshDelay(int refreshDelay) {
        this.refreshDelay = refreshDelay;
    }

}
