package com.bxm.warcar.file.upload.amazon;

import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.bxm.warcar.file.upload.FileException;
import com.bxm.warcar.file.upload.HttpFile;
import com.bxm.warcar.file.upload.HttpFileRequest;
import com.bxm.warcar.file.upload.HttpFileTransverter;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.UUIDHelper;
import org.apache.commons.lang3.StringUtils;

/**
 * <h3>亚马逊s3实现上传文件，返回格式</h3>
 *
 * @author tangxiao
 * @since V1.0.0, 2022/10/13 14:48
 */
public class AmazonHttpFileManager extends HttpFileTransverter {

    private String accessKey;
    private String secretKey;
    private String host;
    private String bucketName;
    private String serviceName;
    private long limitSize;
    private String amazonRegion;


    public AmazonHttpFileManager(String accessKey,
                                 String secretKey,
                                 String host,
                                 String bucketName,
                                 String serviceName,
                                 long limitSize,
                                 String amazonRegion) {
        this.accessKey = accessKey;
        this.secretKey = secretKey;
        this.host = host;
        this.bucketName = bucketName;
        this.serviceName = serviceName;
        this.limitSize = limitSize;
        this.amazonRegion = amazonRegion;
    }

    @Override
	public String upload(HttpFileRequest request)throws Exception{
		HttpFile httpFile =  analysis(request);
		return upload(httpFile);
	}

	private String upload(HttpFile httpFile) throws FileException{
		if (httpFile == null){
			throw new FileException("httpFile is null");
		}

		if (httpFile.getFileSize()> limitSize) {
			throw new FileException("The file is too large ,size is "+httpFile.getFileSize());
		}

		if(StringUtils.isBlank(bucketName)) {
			throw new FileException("bucketName is null");
		}

		String fileName= fileName(httpFile.getFileType());

        AWSCredentials awsCredentials = new BasicAWSCredentials(accessKey, secretKey);

        Regions region =  Regions.fromName(amazonRegion);

        AmazonS3 s3Client = AmazonS3ClientBuilder.standard()
                .withRegion(region)
                .withCredentials(new AWSStaticCredentialsProvider(awsCredentials))
                .build();

        PutObjectRequest request = buildPutObjectRequest(fileName, httpFile);

        try {
            s3Client.putObject(request);
        } finally {
            s3Client.shutdown();
        }
        return StringUtils.join(getHost(),fileName);
	}

    private PutObjectRequest buildPutObjectRequest(String fileName, HttpFile httpFile) {
        ObjectMetadata metadata = new ObjectMetadata();
        metadata.setContentType("text/plain image/jpeg image/png");
        metadata.addUserMetadata("title", "someTitle");

        return new PutObjectRequest(bucketName, fileName, httpFile.getInputStream(), metadata);
    }

    private String fileName(String fileType){
		return StringUtils.join(DateHelper.format("yyyy/MM/dd"),"/",serviceName,"/",UUIDHelper.generate(),fileType);
	}

	private String getHost() {
        if (StringUtils.isNotBlank(host)) {
            return host;
        }
        return "https://" +
                bucketName +
                ".s3.amazonaws.com/";
    }

}
