package com.bxm.warcar.integration.distributed.zk;

import com.bxm.warcar.integration.distributed.DistributedLock;
import com.bxm.warcar.integration.distributed.DistributedLockBus;
import com.bxm.warcar.zk.ZkClientHolder;
import com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.core.MethodIntrospector;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.scheduling.annotation.Schedules;

import java.lang.reflect.Method;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * @author allen
 * @since 1.0.0
 */
public class ZooKeeperDistributedLockBus implements BeanPostProcessor, DistributedLockBus {

    private final static Logger logger = LoggerFactory.getLogger(ZooKeeperDistributedLockBus.class);

    private final ConcurrentMap<String, ZooKeeperDistributedLock> locks = new ConcurrentHashMap<>(64);

    private final ZkClientHolder zkClientHolder;

    public ZooKeeperDistributedLockBus(ZkClientHolder zkClientHolder) {
        Preconditions.checkNotNull(zkClientHolder);
        this.zkClientHolder = zkClientHolder;
    }

    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        return bean;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        Class<?> targetClass = AopUtils.getTargetClass(bean);
        Map<Method, Set<Scheduled>> annotatedMethods = MethodIntrospector.selectMethods(targetClass,
                new MethodIntrospector.MetadataLookup<Set<Scheduled>>() {
                    @Override
                    public Set<Scheduled> inspect(Method method) {
                        Set<Scheduled> scheduledMethods = AnnotatedElementUtils.getMergedRepeatableAnnotations(
                                method, Scheduled.class, Schedules.class);
                        return (!scheduledMethods.isEmpty() ? scheduledMethods : null);
                    }
                });
        if (annotatedMethods.isEmpty()) {
            if (logger.isTraceEnabled()) {
                logger.trace("No @Scheduled annotations found on bean class: " + bean.getClass());
            }
        }
        else {
            // Non-empty set of methods
            for (Map.Entry<Method, Set<Scheduled>> entry : annotatedMethods.entrySet()) {
                Method method = entry.getKey();
                for (Scheduled scheduled : entry.getValue()) {
                    processScheduled(scheduled, method, bean);
                }
            }
            if (logger.isInfoEnabled()) {
                logger.info(annotatedMethods.size() + " @Scheduled methods created distributed lock on bean '" + beanName +
                        "': " + annotatedMethods);
            }
        }
        return bean;
    }

    private void processScheduled(Scheduled scheduled, Method method, Object bean) {
        String path = ZooKeeperDistributedPathUtils.createPath(bean, method);
        ZooKeeperDistributedLock lock = new ZooKeeperDistributedLock(zkClientHolder, path);
        if (locks.containsKey(path)) {
            throw new RuntimeException("Distributed lock '" + path + "' for ZooKeeper already exist!");
        }
        this.locks.putIfAbsent(path, lock);
    }

    @Override
    public DistributedLock getLock(Object bean, Method method) {
        String path = ZooKeeperDistributedPathUtils.createPath(bean, method);
        return this.locks.get(path);
    }
}
