package com.bxm.warcar.integration.message;

import com.bxm.warcar.integration.message.annotation.Messaging;
import com.bxm.warcar.integration.message.failover.SendFailover;
import com.bxm.warcar.mq.Message;
import com.bxm.warcar.mq.Producer;
import com.bxm.warcar.mq.SendResult;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.NamedThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.*;

/**
 * 异步的消息生产者
 *
 * @author allen
 * @since 1.0.0
 */
public class AsyncMessageProducer extends AbstractMessageProducer implements ThreadPoolMessageProducer {

    private static final Logger LOGGER = LoggerFactory.getLogger(AsyncMessageProducer.class);
    private final ExecutorService threadPool;
    private final SendFailover sendFailover;

    public AsyncMessageProducer(Producer producer) {
        this(producer, null);
    }

    public AsyncMessageProducer(Producer producer, Object properties) {
        this(producer, properties, null);
    }

    public AsyncMessageProducer(Producer producer, Object properties, SendFailover sendFailover) {
        this(producer, properties, new ThreadPoolExecutor(10, 10, 0, TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<Runnable>(), new NamedThreadFactory("mp")), sendFailover);
    }

    /**
     * @param producer 消息生产者
     * @param properties 配置对象
     * @param threadPool 线程池，用来异步发送消息
     */
    public AsyncMessageProducer(Producer producer, Object properties, ThreadPoolExecutor threadPool, SendFailover sendFailover) {
        super(producer, properties);
        this.threadPool = threadPool;
        if (null == sendFailover) {
            this.sendFailover = new SendFailover.DefaultSendFailover();
        } else {
            this.sendFailover = sendFailover;
        }
    }

    @Override
    protected void sendMessage(String topic, Object returning, Object arg, Messaging messaging) {
        try {
            threadPool.submit(new MessageSendProcessor(producer, topic, returning, arg, messaging, sendFailover));
        } catch (RejectedExecutionException e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("Rejected: topic={}, arg={}, returning={}", topic, arg, returning);
            }
        }
    }

    @Override
    public ExecutorService getThreadPool() {
        return this.threadPool;
    }

    private static class MessageSendProcessor implements Runnable {

        private final Producer producer;
        private final String topic;
        private final Object returning;
        private final Object arg;
        private final Messaging annotation;
        private final SendFailover sendFailover;

        private MessageSendProcessor(Producer producer, String topic, Object returning, Object arg, Messaging annotation, SendFailover sendFailover) {
            this.producer = producer;
            this.topic = topic;
            this.returning = returning;
            this.arg = arg;
            this.annotation = annotation;
            this.sendFailover = sendFailover;
        }

        @Override
        public void run() {
            sendMessage(returning, arg, annotation);
        }

        private byte[] serialize(Object body) {
            return JsonHelper.convert2bytes(body);
        }

        private void sendMessage(Object returning, Object request, Messaging annotation) {
            String tags = annotation.tags();
            int flag = annotation.flag();
            int delayTimeLevel = annotation.delayTimeLevel();

            MessageBody body = new MessageBody(request, returning);

            Message message = new Message(topic, tags, flag, serialize(body));
            message.setDelayTimeLevel(delayTimeLevel);

            try {
                SendResult result = producer.send(message);
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Message send successful. {}", result.getMsgId());
                }
            } catch (Exception e) {
                if (null != sendFailover) {
                    sendFailover.doFail(message, e);
                }
            }
        }
    }
}
