package com.bxm.warcar.integration.dc.dot.impl;

import com.bxm.warcar.integration.dc.dot.AsyncDot;
import com.bxm.warcar.integration.dc.dot.Dot;
import com.bxm.warcar.integration.dc.dot.DotParameter;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;

import java.util.Map;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 基于线程池的打点实现。
 *
 * @author allen
 * @date 2019/3/7
 * @since 1.0.0
 */
public class ThreadPoolExecutorDotImpl implements DisposableBean, AsyncDot {

    private static final Logger LOGGER = LoggerFactory.getLogger(ThreadPoolExecutorDotImpl.class);
    public static final int DEFAULT_POOL_CORE_SIZE = 10;
    public static final int DEFAULT_AWAIT_TERMINATION_TIME_IN_SECONDS = 50;

    private final Dot dot;
    private final ThreadPoolExecutor pool;
    private final int awaitTerminationTimeInSeconds;

    public ThreadPoolExecutorDotImpl(Dot dot) {
        this(dot, DEFAULT_POOL_CORE_SIZE, DEFAULT_AWAIT_TERMINATION_TIME_IN_SECONDS);
    }

    public ThreadPoolExecutorDotImpl(Dot dot, int poolCoreSize, int awaitTerminationTimeInSeconds) {
        Preconditions.checkNotNull(dot);
        this.dot = dot;
        this.awaitTerminationTimeInSeconds = awaitTerminationTimeInSeconds;
        this.pool = new ThreadPoolExecutor(poolCoreSize, poolCoreSize, 0, TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(), new NamedThreadFactory("dot-pool"));
    }

    @Override
    public void destroy() throws Exception {
        pool.shutdown();
        if (!pool.awaitTermination(awaitTerminationTimeInSeconds, TimeUnit.SECONDS)) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("await termination has timeout!!");
            }
        }
    }

    @Override
    public void asyncDoGet(final DotParameter parameter) {
        this.pool.submit(() -> {
            try {
                dot.doGet(parameter);
            } catch (Exception e) {
                LOGGER.error("doGet: ", e);
            }
        });
    }

    @Override
    public void asyncDotGet(Map<String, Object> parameter) {
        this.pool.submit(() -> {
            try {
                dot.doGet(parameter);
            } catch (Exception e) {
                LOGGER.error("doGet: ", e);
            }
        });
    }

    @Override
    public void asyncDotPost(Map<String, Object> parameter) {
        this.pool.submit(() -> {
            try {
                dot.doPost(parameter);
            } catch (Exception e) {
                LOGGER.error("doPost: ", e);
            }
        });
    }

    @Override
    public void asyncDotPost(String requestJson) {
        this.pool.submit(() -> {
            try {
                dot.doPost(requestJson);
            } catch (Exception e) {
                LOGGER.error("doPost: ", e);
            }
        });
    }

    @Override
    public void asyncDotPostForm(Map<String, Object> parameter) {
        this.pool.submit(() -> {
            try {
                dot.doPostForm(parameter);
            } catch (Exception e) {
                LOGGER.error("doPostForm: ", e);
            }
        });
    }

    @Override
    public int getActiveCount() {
        return this.pool.getActiveCount();
    }

    @Override
    public int getQueueSize() {
        return this.pool.getQueue().size();
    }

    @Override
    public int getCorePoolSize() {
        return this.pool.getCorePoolSize();
    }
}
