package com.bxm.warcar.integration.dc.dot.impl;

import java.time.Duration;
import java.util.Map;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.http.*;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import com.bxm.warcar.integration.dc.dot.Dot;
import com.bxm.warcar.integration.dc.dot.DotConstants;
import com.bxm.warcar.integration.dc.dot.DotParameter;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.base.Preconditions;

/**
 * 默认的打点实现。
 *
 * @author allen
 * @date 2019/3/7
 * @since 1.0.0
 */
public class DefaultDotImpl implements Dot {

    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultDotImpl.class);
    private final String dotUrl;
    private final RestTemplate rest;

    public DefaultDotImpl() {
        this(DotConstants.DEFAULT_DOT_URL);
    }

    public DefaultDotImpl(String dotUrl) {
        Preconditions.checkArgument(StringUtils.isNotBlank(dotUrl));
        this.dotUrl = dotUrl;
        this.rest = new RestTemplateBuilder()
                .setConnectTimeout(Duration.ofSeconds(1))
                .setReadTimeout(Duration.ofSeconds(1))
                .build();
    }

    @Override
    public String doGet(DotParameter parameter) {
        return doGet(parameter, String.class);
    }

    @Override
    public <T> T doGet(DotParameter parameter, Class<T> responseType) {
        return this.doGet0(parameter, responseType);
    }

    @Override
    public String doGet(Map<String, Object> parameter) {
        return doGet(parameter, String.class);
    }

    @Override
    public <T> T doGet(Map<String, Object> parameter, Class<T> responseType) {
        return this.doGet0(parameter, responseType);
    }

    private <T> T doGet0(Map<String, Object> parameter, Class<T> responseType) {
        String url = dotUrl;
        StringBuilder urlConnector = new StringBuilder(StringUtils.removeEnd(dotUrl, "?"));
        if (MapUtils.isNotEmpty(parameter)) {
            urlConnector.append("?");
            for (Map.Entry<String, Object> entry : parameter.entrySet()) {
                Object value = entry.getValue();
                if (null != value) {
                    urlConnector.append(entry.getKey()).append("={").append(entry.getKey()).append("}&");
                }
            }
            url = StringUtils.removeEnd(urlConnector.toString(), "&");
        }

        ResponseEntity<T> responseEntity = rest.getForEntity(url, responseType, parameter);
        HttpStatus statusCode = responseEntity.getStatusCode();
        if (statusCode != HttpStatus.OK) {
            LOGGER.warn("doGet0 code: {} for url: {}", statusCode, url);
        }
        return responseEntity.getBody();
    }

    @Override
    public String doPost(Map<String, Object> parameter) {
        return doPost(parameter, String.class);
    }

    @Override
    public <T> T doPost(Map<String, Object> parameter, Class<T> responseType) {
        return this.doPost0(JsonHelper.convert(parameter), responseType);
    }

    @Override
    public String doPost(String requestJson) {
        return doPost(requestJson, String.class);
    }

    @Override
    public <T> T doPost(String requestJson, Class<T> responseType) {
        return this.doPost0(requestJson, responseType);
    }

    private <T> T doPost0(String requestJson, Class<T> responseType) {
        String url = dotUrl;

        HttpHeaders headers = new HttpHeaders();
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        headers.add("Accept", MediaType.APPLICATION_JSON.toString());
        HttpEntity formEntity = new HttpEntity(requestJson, headers);
        ResponseEntity<T> responseEntity = rest.postForEntity(url, formEntity, responseType);
        HttpStatus statusCode = responseEntity.getStatusCode();
        if (statusCode != HttpStatus.OK) {
            LOGGER.warn("doPost0 code: {} for url: {}", statusCode, url);
        }
        return responseEntity.getBody();
    }

    @Override
    public String doPostForm(Map<String, Object> parameter) {
        return doPostForm(parameter,  String.class);
    }
    
    @Override
    public <T> T doPostForm(Map<String, Object> parameter, Class<T> responseType) {
        MultiValueMap<String, String> postParameter = new LinkedMultiValueMap<>();
        parameter.keySet().forEach(key -> {
            Object value = parameter.get(key);
            if(null != value){
                postParameter.add(key, value.toString());
            }
        });
        return this.doPostForm0(postParameter, responseType);
    }

    private <T> T doPostForm0(MultiValueMap<String, String> parameter, Class<T> responseType) {
        String url = dotUrl;

        HttpHeaders headers = new HttpHeaders();
        MediaType type = MediaType.APPLICATION_FORM_URLENCODED;
        headers.setContentType(type);
        headers.add(org.apache.http.HttpHeaders.ACCEPT, type.toString());
        HttpEntity formEntity = new HttpEntity(parameter, headers);
        ResponseEntity<T> responseEntity = rest.postForEntity(url, formEntity, responseType);
        HttpStatus statusCode = responseEntity.getStatusCode();
        if (statusCode != HttpStatus.OK) {
            LOGGER.warn("doPostForm0 code: {} for url: {}", statusCode, url);
        }
        return responseEntity.getBody();
    }
}
