package com.bxm.warcar.integration.distributed.delayed;

import com.bxm.warcar.utils.LifeCycle;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.redisson.api.RBlockingQueue;
import org.redisson.api.RBucket;
import org.redisson.api.RDelayedQueue;
import org.redisson.api.RedissonClient;

import java.io.Serializable;
import java.time.Duration;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

/**
 * LatestOnlyDelayedQueue<br>
 *
 * 基于 Redisson 的延迟队列工具类，保证对于同一个任务标识（例如 groupId）：<br>
 * - 可以重复添加任务；<br>
 * - 只有最新添加的任务会被执行，旧任务会被惰性丢弃。<br>
 *  <br>
 * 适合分布式环境，消费者在执行任务前，会通过 {@link #latestExecuteTimeProvider}<br>
 * 判断任务是否已经过期或被覆盖，从而避免执行重复或过时的任务。<br>
 *
 * @param <V> 任务的值类型，需实现 Serializable
 * @author Allen
 * @since 2025-9-11
 */
@Slf4j
public class LatestOnlyDelayedTaskQueueTool<V extends Serializable> extends LifeCycle {

    private final RedissonClient redissonClient;
    private final RBlockingQueue<String> blockingQueue;
    private final RDelayedQueue<String> delayedQueue;
    private final ThreadPoolExecutor consumerThread;
    private final Class<V> clazz;
    private final DelayedTaskExecutor<V> executor;
    private final String taskName;
    /**
     * 提供任务最新执行时间的函数
     * <p>
     * 每次消费任务时，会调用此函数获取任务的最新执行时间：
     * - 如果返回 null 或 等于 当前任务执行时间，则任务会被执行；
     * - 否则任务会被惰性丢弃（说明已经有更新的任务替代它）。
     */
    private final Function<V, Long> latestExecuteTimeProvider;

    private final ObjectMapper objectMapper = new ObjectMapper();

    private final Duration timeToLive;

    /**
     * @param redissonClient redisson客户端
     * @param taskName       任务名称，要求唯一性
     * @param executor       任务消费者，当延迟时间到达时被调用
     * @param latestExecuteTimeProvider 任务当前执行时间获取器，用于判断任务是否已过期，请根据任务值返回任务最新的执行时间。
     *                                  如果返回 null，任务会被立即执行；
     *                                  如果返回的时间等于任务执行时间，任务会被立即执行；
     *                                  否则任务不会被消费者执行。
     */
    public LatestOnlyDelayedTaskQueueTool(RedissonClient redissonClient, String taskName, Class<V> clazz, DelayedTaskExecutor<V> executor, Function<V, Long> latestExecuteTimeProvider) {
        this(redissonClient, taskName, clazz, executor, latestExecuteTimeProvider, Duration.ofDays(1));
    }

    /**
     * @param redissonClient redisson客户端
     * @param taskName       任务名称，要求唯一性
     * @param executor       任务消费者，当延迟时间到达时被调用
     * @param latestExecuteTimeProvider 任务当前执行时间获取器，用于判断任务是否已过期，请根据任务值返回任务最新的执行时间。
     *                                  如果返回 null，任务会被立即执行；
     *                                  如果返回的时间等于任务执行时间，任务会被立即执行；
     *                                  否则任务不会被消费者执行。
     * @param timeToLive 在此时间内，任务保证只会被执行一次。
     */
    public LatestOnlyDelayedTaskQueueTool(RedissonClient redissonClient, String taskName, Class<V> clazz, DelayedTaskExecutor<V> executor, Function<V, Long> latestExecuteTimeProvider, Duration timeToLive) {
        this.redissonClient = redissonClient;
        this.clazz = clazz;
        this.taskName = taskName;
        this.blockingQueue = redissonClient.getBlockingQueue(taskName);
        this.delayedQueue = redissonClient.getDelayedQueue(blockingQueue);
        this.consumerThread = new ThreadPoolExecutor(1, 1, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<>(), new NamedThreadFactory(String.format("delayed-%s-queue-tool", taskName)));
        this.executor = executor;
        this.latestExecuteTimeProvider = latestExecuteTimeProvider;
        this.timeToLive = timeToLive;
    }

    /**
     * 添加延迟任务
     * <p>
     * 注意：
     * - 可以重复添加相同任务；
     * - 实际消费时只有最新任务会被执行，旧任务会被惰性丢弃。
     *
     * @param delayedTask 待添加的延迟任务
     */
    public void offer(DelayedTask<V> delayedTask) {
        try {
            Long executeTimeInMillis = delayedTask.getExecuteTimeInMillis();
            long delay = executeTimeInMillis - System.currentTimeMillis();
            if (delay < 0) {
                log.warn("delayed task execute time is in the past, task: {}", delayedTask);
                return;
            }
            String serialize = serialize(delayedTask);
            delayedQueue.offer(serialize, delay, TimeUnit.MILLISECONDS);
        } catch (JsonProcessingException e) {
            log.error("serialize delayed task error, task: {}", delayedTask, e);
        }
    }

    /**
     * 清空延迟队列（慎用）
     */
    public void clear() {
        delayedQueue.clear();
        blockingQueue.clear();
        log.warn("{} delayed queue tool cleared", taskName);
    }

    /**
     * 获取当前延迟队列大小
     */
    public int size() {
        return delayedQueue.size();
    }

    private String serialize(DelayedTask<V> delayedTask) throws JsonProcessingException {
        return objectMapper.writeValueAsString(delayedTask);
    }

    private DelayedTask<V> deserialize(String value) throws JsonProcessingException {
        JavaType javaType = objectMapper.getTypeFactory()
                .constructParametricType(DelayedTask.class, clazz);
        return objectMapper.readValue(value, javaType);
    }

    @Override
    protected void doInit() {
        if (null == executor) {
            log.info("delayed {} queue tool executor is null", taskName);
            return;
        }
        consumerThread.execute(() -> {
            while (true) {
                try {
                    String value = blockingQueue.take();
                    DelayedTask<V> delayedTask = deserialize(value);
                    Long latestExecuteTime = latestExecuteTimeProvider.apply(delayedTask.getValue());
                    Long taskExecuteTime = delayedTask.getExecuteTimeInMillis();
                    if (null == latestExecuteTime || latestExecuteTime.equals(taskExecuteTime)) {
                        if (isFirstExecute(delayedTask)) {
                            log.info("{} DelayedTask [{}] will be executed, taskExecuteTime: {}, currentExecuteTime: {}", taskName, delayedTask.getValue(), taskExecuteTime, latestExecuteTime);
                            executor.accept(delayedTask);
                        } else {
                            log.debug("{} DelayedTask [{}] already executed, skip, taskExecuteTime: {}, but currentExecuteTime is: {}", taskName, delayedTask.getValue(), taskExecuteTime, latestExecuteTime);
                        }
                    } else {
                        log.debug("{} DelayedTask [{}] was expired, taskExecuteTime: {}, but currentExecuteTime is: {}", taskName, delayedTask.getValue(), taskExecuteTime, latestExecuteTime);
                        executor.reject(delayedTask);
                    }
                } catch (InterruptedException ignored) {
                    Thread.currentThread().interrupt();
                    break;
                } catch (Exception e) {
                    log.error("delayed {} queue tool consume error", taskName, e);
                    // backoff 防止无限打日志
                    try {
                        Thread.sleep(1000);
                    } catch (InterruptedException ignored) {
                        Thread.currentThread().interrupt();
                    }
                }
            }
        });
        log.info("delayed {} queue tool init success", taskName);
    }

    @Override
    protected void doDestroy() {
        consumerThread.shutdownNow();
        log.info("delayed {} queue tool shutdown success", taskName);
    }

    /**
     * 判断是否首次执行该任务（幂等控制）
     */
    private boolean isFirstExecute(DelayedTask<V> delayedTask) {
        try {
            String taskKey = DigestUtils.md5Hex(objectMapper.writeValueAsString(delayedTask.getValue()));
            String key = String.format("warcar:delayed_queue:execute:%s:%s:%d",
                    taskName, taskKey, delayedTask.getExecuteTimeInMillis());
            RBucket<Integer> bucket = redissonClient.getBucket(key);
            long seconds = timeToLive.getSeconds();
            return bucket.trySet(1, seconds, TimeUnit.SECONDS);
        } catch (Exception e) {
            log.error("generate execute flag error, fallback execute task: {}", delayedTask, e);
            // 异常情况下默认执行，保证任务不丢
            return true;
        }
    }

}
