package com.bxm.warcar.integration.eventbus;

import com.bxm.warcar.integration.eventbus.core.AsyncEventBus;
import com.bxm.warcar.integration.eventbus.core.EventBus;
import com.bxm.warcar.utils.NamedThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;

import java.util.EventObject;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * <p>异步的事件注册、提交中心</p>
 *
 * <p>异步队列，不会阻塞发送 {@link #post(EventObject)} 线程，生产的消息会先 {@code add} 到 {@code Dispatcher#ConcurrentLinkedQueue} 队列里，然后 {@code pool} 队列，使用线程池处理。
 * <p>消费者执行时会占用线程池队列，声明 @AllowConcurrentEvents 表示这个监听器支持多线程处理，否则同一时间只会处理一条消息。
 *
 * <p>如果没有监听者将发送死信事件 DeadEvent
 *
 * @author allen
 * @since V1.0.0 2017/12/15
 */
public final class AsyncEventPark extends AbstractEventPark implements DisposableBean {

    private static final Logger LOGGER = LoggerFactory.getLogger(AsyncEventPark.class);
    private static final int DEFAULT_AWAIT_TERMINATION_SECONDS = 30;
    private final AsyncEventBus asyncEventBus;
    private final ThreadPoolExecutor threadPoolExecutor;
    private final int awaitTerminationSeconds;

    public AsyncEventPark() {
        this(Runtime.getRuntime().availableProcessors() * 2 + 1, DEFAULT_AWAIT_TERMINATION_SECONDS);
    }

    public AsyncEventPark(int coreSize) {
        this(coreSize, DEFAULT_AWAIT_TERMINATION_SECONDS);
    }

    public AsyncEventPark(int coreSize, int awaitTerminationSeconds) {
        this(new ThreadPoolExecutor(coreSize, coreSize, 0, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<>(), new NamedThreadFactory("event-park")),
                awaitTerminationSeconds);
    }

    public AsyncEventPark(ThreadPoolExecutor executor) {
        this(executor, DEFAULT_AWAIT_TERMINATION_SECONDS);
    }

    public AsyncEventPark(ThreadPoolExecutor executor, int awaitTerminationSeconds) {
        this.threadPoolExecutor = executor;
        this.asyncEventBus = new AsyncEventBus("async-event-bus", executor);
        this.awaitTerminationSeconds = awaitTerminationSeconds;
        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("ThreadPoolExecutor current core pool size is {}", executor.getCorePoolSize());
        }
    }

    @Override
    protected EventBus getEventBus() {
        return asyncEventBus;
    }

    @Override
    protected String getEventParkName() {
        return "AsyncEventPark";
    }

    public int getCorePoolSize() {
        return this.threadPoolExecutor.getCorePoolSize();
    }

    public int getActiveCount() {
        return this.threadPoolExecutor.getActiveCount();
    }

    public int getQueueSize() {
        return this.threadPoolExecutor.getQueue().size();
    }

    @Override
    public void destroy() throws Exception {
        this.threadPoolExecutor.shutdown();
        if (!this.threadPoolExecutor.awaitTermination(awaitTerminationSeconds, TimeUnit.SECONDS)) {
            LOGGER.warn("ThreadPoolExecutor {} terminated fail", this.threadPoolExecutor);
        } else {
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("ThreadPoolExecutor {} terminated success", this.threadPoolExecutor);
            }
        }
    }
}
