package com.bxm.warcar.logging.support;

import com.bxm.warcar.logging.Logging;
import com.bxm.warcar.logging.LoggingWriter;
import com.bxm.warcar.utils.NamedThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.data.mongodb.core.MongoTemplate;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 日志写入到 MongoDB 中。队列生产，批量消费并批量插入到数据库中。不会阻塞主线程。
 * @author allen
 * @since 2020-03-19
 */
public class MongodbLoggingWriter implements LoggingWriter, DisposableBean {

    private static final Logger LOGGER = LoggerFactory.getLogger(MongodbLoggingWriter.class);
    private final MongoTemplate mongoTemplate;
    private final LinkedBlockingQueue<Logging> queue;
    private final ScheduledExecutorService consumer;

    public MongodbLoggingWriter(MongoTemplate mongoTemplate) {
        this.mongoTemplate = mongoTemplate;

        this.queue = new LinkedBlockingQueue<>();
        this.consumer = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("queue-consumer"));
        this.startConsumer();
    }

    @Override
    public void destroy() throws Exception {
        this.consumer.shutdown();
        this.consumer.awaitTermination(5, TimeUnit.SECONDS);
    }

    private void startConsumer() {
        this.consumer.scheduleWithFixedDelay(() -> {
            try {
                List<Logging> dest = new ArrayList<>(100);
                queue.drainTo(dest, 100);
                mongoTemplate.insert(dest, Logging.class);
            } catch (Exception e) {
                LOGGER.error("", e);
            }
        }, 10, 5, TimeUnit.SECONDS);
    }

    @Override
    public void write(Logging logging) {
        while (!queue.offer(logging)) {
            try {
                // 装不下，队列头部丢弃一条
                queue.take();
            } catch (InterruptedException e) {
                // no-op;
            }
        }
    }
}
