package com.bxm.warcar.micrometer.ip;

import com.bxm.warcar.ip.IpLibrary;
import com.bxm.warcar.utils.MapHelper;
import com.google.common.collect.Maps;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Timer;
import io.micrometer.core.instrument.binder.MeterBinder;
import org.springframework.util.ClassUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;

/**
 * @author allen
 * @date 2021-04-28
 * @since 1.0
 */
public class IpLibraryMeter implements MeterBinder {

    private final Map<String, Timer> timer = Maps.newHashMap();
    private final Map<String, Gauge> gauge = Maps.newHashMap();
    /**
     * 总识别次数
     */
    private final Map<String, AtomicLong> total = Maps.newHashMap();
    /**
     * 识别成功次数
     */
    private final Map<String, AtomicLong> success = Maps.newHashMap();

    private final Iterable<IpLibrary> ipLibraryIterator;

    public IpLibraryMeter(Iterable<IpLibrary> ipLibraryIterator) {
        this.ipLibraryIterator = ipLibraryIterator;
    }

    @Override
    public void bindTo(MeterRegistry registry) {
        ipLibraryIterator.forEach((e) -> {
            String simpleName = getSimpleName(e);
            timer.put(simpleName, Timer.builder("warcar.iplibrary.time").tags("name", simpleName).register(registry));
            Gauge.builder("warcar.iplibrary.gauge", new Supplier<Number>() {
                @Override
                public Number get() {
                    AtomicLong t = total.get(simpleName);
                    AtomicLong s = success.get(simpleName);
                    if (Objects.isNull(t)) {
                        return 0L;
                    }
                    return new BigDecimal(Optional.ofNullable(s).orElse(new AtomicLong(0L)).get())
                            .divide(BigDecimal.valueOf(t.get()),
                                    3, RoundingMode.HALF_UP);
                }
            }).tags("name", simpleName).register(registry);
        });
    }

    private String getSimpleName(IpLibrary e) {
        return ClassUtils.getUserClass(e).getSimpleName();
    }

    Timer getTimer(IpLibrary instance) {
        return timer.get(getSimpleName(instance));
    }

    void incrementSuccess(IpLibrary e) {
        MapHelper.get(success, getSimpleName(e), new AtomicLong(0L)).incrementAndGet();
    }

    void incrementTotal(IpLibrary e) {
        MapHelper.get(total, getSimpleName(e), new AtomicLong(0L)).incrementAndGet();
    }
}
