package com.bxm.warcar.mq.rocketmq;

import com.bxm.warcar.mq.ConsumeStatus;
import com.bxm.warcar.mq.Consumer;
import com.bxm.warcar.mq.Message;
import com.bxm.warcar.mq.SingleMessageListener;
import com.bxm.warcar.utils.LifeCycle;
import com.google.common.base.Preconditions;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.consumer.listener.ConsumeConcurrentlyContext;
import org.apache.rocketmq.client.consumer.listener.ConsumeConcurrentlyStatus;
import org.apache.rocketmq.client.consumer.listener.MessageListenerConcurrently;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.common.message.MessageExt;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 基于RocketMQ的实现。
 * 参考资料：<a href="https://rocketmq.apache.org">https://rocketmq.apache.org</a>
 * @author allen
 */
public class RocketmqConsumer extends LifeCycle implements Consumer {

    private static final Logger LOGGER = LoggerFactory.getLogger(RocketmqConsumer.class);
    private final DefaultMQPushConsumer consumer;
    private final AtomicBoolean started = new AtomicBoolean(false);
    private final String namesrvAddr;
    private SingleMessageListener messageListener;
    private final String consumerGroup;
    private String subExpression;

    public RocketmqConsumer(String namesrvAddr, SingleMessageListener messageListener) {
        this(namesrvAddr, messageListener, Consumer.DEFAULT_CONSUMER_GROUP);
    }

    public RocketmqConsumer(String namesrvAddr, SingleMessageListener messageListener, String consumerGroup) {
        super();
        this.namesrvAddr = namesrvAddr;
        this.messageListener = messageListener;
        this.consumerGroup = consumerGroup;
        this.consumer = new DefaultMQPushConsumer(consumerGroup);
        this.consumer.setNamesrvAddr(namesrvAddr);
        if (StringUtils.isBlank(subExpression)) {
            this.subExpression = "*";
        }
    }

    @Override
    protected void doInit() {
        try {
            Preconditions.checkArgument(StringUtils.isNotBlank(messageListener.getTopic()), "topic must has not blank!");
            consumer.subscribe(messageListener.getTopic(), subExpression);
            consumer.registerMessageListener(new MessageListenerConcurrently() {
                
                @Override
                public ConsumeConcurrentlyStatus consumeMessage(List<MessageExt> msgs, ConsumeConcurrentlyContext context) {
                    if (CollectionUtils.isEmpty(msgs)) {
                        return ConsumeConcurrentlyStatus.CONSUME_SUCCESS;
                    }
                    if (msgs.size() > 1) {
                        if (LOGGER.isWarnEnabled()) {
                            LOGGER.warn("Unsupported batch messages!!");
                        }
                        return ConsumeConcurrentlyStatus.RECONSUME_LATER;
                    }

                    Message msg = new Message();
                    MessageExt messageExt = msgs.get(0);
                    msg.setMsgId(messageExt.getMsgId());
                    msg.setBody(messageExt.getBody());
                    msg.setTags(messageExt.getTags());
                    msg.setTopic(messageExt.getTopic());
                    msg.setFlag(messageExt.getFlag());

                    ConsumeStatus status = messageListener.consume(msg, context);
                    switch (status) {
                        case CONSUME_SUCCESS:
                            return ConsumeConcurrentlyStatus.CONSUME_SUCCESS;

                        default:
                            return ConsumeConcurrentlyStatus.RECONSUME_LATER;
                    }
                }
            });
            consumer.start();
            started.set(true);
        }
        catch (MQClientException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    protected void doDestroy() {
        shutdown();
        started.set(false);
    }

    @Override
    public void suspend(){
        if (null != consumer) {
            consumer.suspend();
        }
    }

    @Override
    public void shutdown() {
        if (null != consumer) {
            consumer.shutdown();
        }
    }

    @Override
    public void start() {
        doInit();
    }

    @Override
    public boolean isStarted() {
        return started.get();
    }

    public String getConsumerGroup() {
        return consumerGroup;
    }

    public String getSubExpression() {
        return subExpression;
    }

    public void setSubExpression(String subExpression) {
        this.subExpression = subExpression;
    }

    public DefaultMQPushConsumer getConsumer() {
        return consumer;
    }

    public String getNamesrvAddr() {
        return namesrvAddr;
    }

    @Override
    public SingleMessageListener getMessageListener() {
        return messageListener;
    }

    public void setMessageListener(SingleMessageListener messageListener) {
        this.messageListener = messageListener;
    }
}
