package com.bxm.warcar.mq.redis;

import com.bxm.warcar.mq.Consumer;
import com.bxm.warcar.mq.Listener;
import com.bxm.warcar.mq.Message;
import com.bxm.warcar.mq.SingleMessageListener;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.LifeCycle;
import com.bxm.warcar.utils.NamedThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.util.List;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @author allen
 * @date 2019/10/21
 * @since 1.0.0
 */
public class JedisConsumer extends LifeCycle implements Consumer {

    private static final Logger LOGGER = LoggerFactory.getLogger(RedisConsumer.class);
    private final ThreadPoolExecutor consumer = new ThreadPoolExecutor(1, 1, 0, TimeUnit.SECONDS,
            new LinkedBlockingQueue<>(), new NamedThreadFactory("JedisConsumer"));
    private final JedisPool jedisPool;
    private final SingleMessageListener messageListener;
    private final AtomicBoolean shutdown = new AtomicBoolean(false);

    public JedisConsumer(JedisPool jedisPool, SingleMessageListener messageListener) {
        this.jedisPool = jedisPool;
        this.messageListener = messageListener;
    }


    @Override
    protected void doInit() {
        String topic = messageListener.getTopic();
        consumer.execute(() -> {
            for (;;) {
                if (shutdown.get()) {
                    break;
                }
                if (null == jedisPool || jedisPool.isClosed()) {
                    try {
                        Thread.sleep(1000L);
                    } catch (InterruptedException ignore) {
                    }
                    continue;
                }
                Jedis jedis = null;
                jedis = jedisPool.getResource();
                try {
                    List<String> list = jedis.blpop(1, RedisConst.key(topic));
                    for (int index = 0; index < list.size(); index++) {
                        if (index == 0) {
                            // The first value is key name.
                            continue;
                        }
                        String json = list.get(index);
                        Message message = JsonHelper.convert(json, Message.class);
                        if (null != message) {
                            messageListener.consume(message, jedis);
                        }
                    }
                } catch (Exception e) {
                    LOGGER.error("", e);
                } finally {
                    if (null != jedis) {
                        jedis.close();
                    }
                }
            }
        });
    }

    @Override
    protected void doDestroy() {
        consumer.shutdown();
        shutdown.compareAndSet(false, true);
        try {
            consumer.awaitTermination(5, TimeUnit.SECONDS);
        } catch (InterruptedException ignore) {
        }
    }

    @Override
    public void suspend() {

    }

    @Override
    public void shutdown() {
        this.doDestroy();
    }

    @Override
    public void start() {
        this.doInit();
    }

    @Override
    public boolean isStarted() {
        return false;
    }

    @Override
    public Listener getMessageListener() {
        return messageListener;
    }
}
