package com.bxm.warcar.sc.domain;

import com.bxm.warcar.integration.pair.Pair;
import com.bxm.warcar.utils.PathUtils;
import com.google.common.base.Preconditions;
import org.apache.commons.beanutils.MethodUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.reflect.FieldUtils;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;

import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.util.Map;

/**
 * 静态资源域名替换处理器
 *
 * @see StaticResource
 * @see StaticResourceMapping
 * @author allen
 * @date 2019/1/21
 * @since 1.0.0
 */
@Aspect
public class StaticResourceReplaceHandler {

    private static final String DOMAIN_KEY = "STATIC_RESOURCE_DOMAIN";
    private final Pair pair;

    public StaticResourceReplaceHandler(Pair pair) {
        Preconditions.checkNotNull(pair);
        this.pair = pair;
    }

    @Pointcut("@annotation(com.bxm.warcar.sc.domain.StaticResourceMapping)")
    public void pointcut() {}

    @AfterReturning(value = "pointcut()", returning = "returning")
    public void afterRetuning(Object returning) {
        if (null == returning) {
            return;
        }
        if (Iterable.class.isAssignableFrom(returning.getClass())) {
            this.replaceRootForIterable(returning);
        } else if (returning.getClass().isArray()) {
            this.replaceRootForArray(returning);
        } else {
            this.replaceForObject(returning);
        }
    }

    private void replaceRootForIterable(Object retuning) {
        Iterable iterable = (Iterable) retuning;
        for (Object o : iterable) {
            this.replaceForObject(o);
        }
    }
    private void replaceRootForArray(Object retuning) {
        Object[] array = (Object[]) retuning;
        for (Object o : array) {
            this.replaceForObject(o);
        }
    }

    private void replaceObject(Object retuning) {
        Class<?> cls = retuning.getClass();
        while (null != cls) {
            Field[] declaredFields = cls.getDeclaredFields();
            for (Field declaredField : declaredFields) {

                String fieldName = declaredField.getName();
                Class<?> fieldClass = declaredField.getType();

                Object value = getObject(retuning, fieldName);
                if (null == value) {
                    continue;
                }

                if (value instanceof Number){
                    continue;
                }

                if (value instanceof Boolean){
                    continue;
                }

                if (value instanceof String) {
                    this.replaceForString(retuning, fieldName);
                    continue;
                }
                if (value instanceof Map) {
                    this.replaceForMap(value);
                    continue;
                }

                if (value instanceof Iterable) {
                    this.replaceForIterable(value);
                    continue;
                }

                if (value instanceof Array) {
                    this.replaceForArray(value);
                    continue;
                }

                //if (fieldClass.getClassLoader()!=null) {
                //}
                if (Object.class.isAssignableFrom(fieldClass)){

                    this.replaceObject(value);
                }
            }
            cls = cls.getSuperclass();
        }
    }

    private  void replaceForObject(Object value){
        if (null == value) {
            return;
        }

        if (value instanceof Number){
            return;
        }

        if (value instanceof Boolean){
            return;
        }
        this.replaceObject(value);
    }

    private void replaceForArray(Object value) {
        Array[] array = (Array[]) value;
        for (Object o : array) {
            this.replaceForObject(o);
        }
    }

    private void replaceForIterable(Object value) {
        Iterable<?> iterable = (Iterable) value;
        for (Object o : iterable) {
            this.replaceForObject(o);
        }
    }

    private void replaceForMap(Object value) {
        disposeMap(value);
    }

    private void  disposeMap(Object value){
        Map<Object, Object> map = (Map) value;
        for (Map.Entry<?, ?> entry : map.entrySet()) {
            Object mv = entry.getValue();
            if(mv == null){
               continue;
            }
            if (mv instanceof String) {
                String url = replace(mv.toString());
                Object key = entry.getKey();
                map.put(key, url);
            }else if(Object.class.isAssignableFrom(mv.getClass())){
                replaceForObject(mv);
            }
        }
    }

    private void replaceForString(Object retuning, String fieldName) {
        Object value = getObject(retuning, fieldName);

        if (null == value) {
            return;
        }
        String newUrl = replace(value.toString());

        try {
            FieldUtils.writeField(retuning, fieldName, newUrl, true);
        } catch (IllegalAccessException | IllegalArgumentException e) {
            this.setValue(retuning, fieldName, newUrl);
        }
    }

    private Object getObject(Object retuning, String fieldName) {
        Object value;
        try {
            value = FieldUtils.readDeclaredField(retuning, fieldName, true);
        } catch (Exception e) {
            value = getValue(retuning, fieldName);
        }
        return value;
    }

    private Object getValue(Object object, String fieldName) {
        String methodName = StringUtils.join(new String[]{"get", StringUtils.capitalize(fieldName)});
        try {
            return MethodUtils.invokeMethod(object, methodName, new Object[0]);
        } catch (Exception e) {
            return null;
        }
    }

    private void setValue(Object object, String fieldName, Object value) {
        String methodName = StringUtils.join(new String[]{"set", StringUtils.capitalize(fieldName)});
        try {
            MethodUtils.invokeMethod(object, methodName, value);
        } catch (Exception ignored) {
        }
    }

    private String replace(String url) {
        ResourceModel resourceModel = pair.get(DOMAIN_KEY).toObject(ResourceModel.class);

        if (resourceModel == null ){
            return url;
        }

        if ( CollectionUtils.isEmpty(resourceModel.getSourceList())
                ||  StringUtils.isBlank(resourceModel.getTargetUrl())){
            return url;
        }

        if (resourceModel.isContains(url)){
            return PathUtils.replaceDomain(url, resourceModel.getTargetUrl());
        }

        return url;
    }
}
