package com.bxm.warcar.sentinel;

import com.alibaba.csp.sentinel.concurrent.NamedThreadFactory;
import com.alibaba.csp.sentinel.datasource.AbstractDataSource;
import com.alibaba.csp.sentinel.datasource.Converter;
import com.bxm.warcar.utils.StringHelper;
import com.bxm.warcar.zk.ZkClientHolder;
import com.bxm.warcar.zk.listener.NodeChangedListener;
import com.bxm.warcar.zk.listener.node.NodeChanged;
import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.curator.framework.CuratorFramework;
import org.apache.zookeeper.data.Stat;

import java.util.Collection;
import java.util.Objects;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @author allen
 * @date 2022-03-29
 * @since 1.0
 */
@Slf4j
public class ZooKeeperDataSource<T> extends AbstractDataSource<byte[], T> {

    private final ExecutorService pool = new ThreadPoolExecutor(1, 1, 0, TimeUnit.MILLISECONDS,
            new ArrayBlockingQueue<Runnable>(1), new NamedThreadFactory("sentinel-zookeeper-ds-update"),
            new ThreadPoolExecutor.DiscardOldestPolicy());

    private final String path;

    private ZkClientHolder zkClientHolder;
    private NodeChanged nodeChanged;
    private NodeChangedListener listener;

    public ZooKeeperDataSource(final ZkClientHolder zkClientHolder, final String groupId, final String dataId, Converter<byte[], T> parser) {
        super(parser);
        Preconditions.checkNotNull(zkClientHolder);
        Preconditions.checkArgument(StringUtils.isNotBlank(groupId));
        Preconditions.checkArgument(StringUtils.isNotBlank(dataId));
        this.zkClientHolder = zkClientHolder;
        this.path = getPath(groupId, dataId);
        init();
    }

    private void init() {
        creatingPathIfNeeded();
        initZookeeperListener();
    }

    private void creatingPathIfNeeded() {
        try {
            CuratorFramework zkClient = zkClientHolder.get();
            Stat stat = zkClient.checkExists().forPath(path);
            if (Objects.isNull(stat)) {
                String forPath = zkClient.create().creatingParentsIfNeeded().forPath(path, StringHelper.convert("{}"));
                log.info(String.format("[ZookeeperDataSource] create path %s", forPath));
            }
        } catch (Exception ex) {
            log.warn("[ZookeeperDataSource] Error when creating path", ex);
        }
    }

    private void initZookeeperListener() {
        try {
            this.listener = new NodeChangedListener() {
                @Override
                public void update(String path, byte[] data) {
                    T newValue = parser.convert(data);
                    if (Objects.isNull(newValue)) {
                        return;
                    }
                    log.info("[ZookeeperDataSource] New property value received for ({}): {}", path, newValue);
                    getProperty().updateValue(newValue);
                }

                @Override
                public ExecutorService getExecutorService() {
                    return pool;
                }
            };

            this.nodeChanged = new NodeChanged(zkClientHolder, path, listener);
            this.nodeChanged.init();
        } catch (Exception e) {
            log.warn("[ZookeeperDataSource] Error occurred when initializing Zookeeper data source", e);
        }
    }

    @Override
    public byte[] readSource() throws Exception {
        return zkClientHolder.get().getData().forPath(path);
    }

    @Override
    public void close() throws Exception {
        if (Objects.nonNull(nodeChanged)) {
            nodeChanged.destroy();
        }
        pool.shutdown();
    }

    private String getPath(String groupId, String dataId) {
        return String.format("/%s/%s", groupId, dataId);
    }
}
