package com.bxm.warcar.utils.crypto;

import lombok.AllArgsConstructor;
import lombok.Builder;

import java.util.function.Function;

/**
 * 输入输出都是 String 的AES加解密。
 *
 * @author allen
 * @date 2021-08-26
 * @since 1.0
 */
@Builder
@AllArgsConstructor
public class StringAesCipher {

    /**
     * 输入 String 类型的key，序列化成 byte[] 的实现。
     */
    private final Function<String, byte[]> keySerializer;
    /**
     * 加密时，输入 String 类型的明文 content，序列化成 byte[] 的实现。
     */
    private final Function<String, byte[]> contentEncryptSerializer;
    /**
     * 加密时，输出 byte[] 类型的密文 content，序列化成 String 的实现。
     */
    private final Function<byte[], String> contentEncryptDeserializer;
    /**
     * 解密时，输入 String 类型的密文 content，序列化成 byte[] 的实现。
     */
    private final Function<String, byte[]> contentDecryptSerializer;
    /**
     * 解密时，输出 byte[] 类型的明文 content，序列化成 String 的实现。
     */
    private final Function<byte[], String> contentDecryptDeserializer;

    public String encrypt(String key, String content) throws Exception {
        byte[] k = keySerializer.apply(key);
        byte[] v = contentEncryptSerializer.apply(content);
        byte[] b = AesCipher.encrypt(k, v);
        return contentEncryptDeserializer.apply(b);
    }

    public String decrypt(String key, String content) throws Exception {
        byte[] k = keySerializer.apply(key);
        byte[] v = contentDecryptSerializer.apply(content);
        byte[] b = AesCipher.decrypt(k, v);
        return contentDecryptDeserializer.apply(b);
    }
}
