
package com.bxm.warcar.utils.http;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.util.Map;

/**
 * <h3>基于httpClien  http工具类</h3>
 * <p>
 *     支持 http GET , POST
 *     支持body,header传参
 *     线程池请求{@link com.bxm.warcar.utils.http.pool.RestTemplatePool}
 *     其他http请求工具类{@link OkHttpUtils,HttpClientUtils}
 * </p>
 *
 * @author hcmony
 * @since V1.0.0, 2018/01/30 10:14
 */
public class RestTemplateUtils {

    private static RestTemplate restTemplate = new RestTemplate();

    /**
     * get 请求url后面拼接参数
     *
     * @param url http://api.bianxiao.com?test=1
     * @return
     */
    public static String get(String url) {
        return exchange(url, HttpMethod.GET).getBody();
    }

    /**
     * get 请求url后面拼接参数
     *
     * @param url     http://api.bianxiao.com?test=1
     * @param headers heders传参
     * @return
     */
    public static String get(String url, Map<String, String> headers) {
        return get(url, Maps.newHashMap(), headers);
    }

    /**
     * get 请求
     *
     * @param url     http://api.bianxiao.com
     * @param params  params传参
     * @param headers heders传参
     * @return
     */
    public static String get(String url, Map<String, Object> params, Map<String, String> headers) {
        return exchange(url, headers, params, HttpMethod.GET, String.class).getBody();
    }

    /**
     * post 请求
     *
     * @param url http://api.bianxiao.com
     * @return
     */
    public static String post(String url) {
        return exchange(url, HttpMethod.POST).getBody();
    }

    /**
     * post 请求
     *
     * @param url    http://api.bianxiao.com
     * @param params params传参
     * @return
     */
    public static String post(String url, Map<String, Object> params) {
        return post(url, params, Maps.newHashMap());
    }

    /**
     * post 请求
     *
     * @param url     http://api.bianxiao.com
     * @param params  params传参
     * @param headers heders传参
     * @return
     */
    public static String post(String url, Map<String, Object> params, Map<String, String> headers) {
        return exchange(url, headers, params, HttpMethod.POST, String.class).getBody();
    }

    /**
     * post 请求 支持@RequestBody 传参请求
     *
     * @param url     http://api.bianxiao.com
     * @param o       实体传参
     * @param headers heders传参
     * @return
     */
    public static String postRequestBody(String url, Object o, Map<String, String> headers) {
        return exchange(url, headers, o, HttpMethod.POST, String.class).getBody();
    }

    /**
     *
     * @param url http://api.bianxiao.com
     * @param method HttpMethod.POST,HttpMethod.GET {@link HttpMethod}
     * @return
     */
    public static HttpEntity<String> exchange(String url, HttpMethod method) {
        return exchange(url, method, Maps.newHashMap());
    }

    /**
     *
     * @param url http://api.bianxiao.com
     * @param responseType String.class
     * @return
     */
    public static <T> HttpEntity<T> get(String url, Class<T> responseType) {
        return exchange(url, HttpMethod.GET, responseType);
    }

    /**
     *
     * @param url http://api.bianxiao.com
     * @param responseType String.class
     * @return
     */
    public static <T> HttpEntity<T> post(String url, Class<T> responseType) {
        return exchange(url, HttpMethod.POST, responseType);
    }

    /**
     *
     * @param url http://api.bianxiao.com
     * @param method HttpMethod.POST,HttpMethod.GET {@link HttpMethod}
     * @param responseType String.class
     * @param <T>
     * @return
     */
    public static <T> HttpEntity<T> exchange(String url, HttpMethod method, Class<T> responseType) {
        return exchange(url, Maps.newHashMap(), method, responseType);
    }

    public static HttpEntity<String> exchange(String url, HttpMethod method, Map<String, Object> bodyParams) {
        return exchange(url, bodyParams, method, String.class);
    }

    public static <T> HttpEntity<T> exchange(String url, Map<String, Object> bodyParams, HttpMethod method, Class<T> responseType) {
        return exchange(url, Maps.newHashMap(), bodyParams, method, responseType);
    }

    public static  <T> HttpEntity<T> exchange(String url, Map<String, String> headers,Object bodyParams, HttpMethod method, Class<T> responseType) {
        return exchange(url, headers, bodyParams,method, responseType,null);
    }

    public static <T> HttpEntity<T> exchange(String url, Map<String, String> headers,Object bodyParams, HttpMethod method, Class<T> responseType,String contentType) {
        if (StringUtils.isNotBlank(contentType)){
            headers.put("Content-Type",contentType);
        }
        HttpEntity httpEntity = new HttpEntity<>(bodyParams, RestTemplateUtils.headerParams(headers));
        if (bodyParams instanceof Map){
            MultiValueMap multiValueMap = RestTemplateUtils.praseBodyParams((Map<String, Object>)bodyParams);
            httpEntity = new HttpEntity<>(multiValueMap, RestTemplateUtils.headerParams(headers));
            return restTemplate.exchange(url, method, httpEntity, responseType);
        }
        return restTemplate.exchange(url, method, httpEntity, responseType);
    }

    public static LinkedMultiValueMap<String, String> headerParams(Map<String, String> headers){
        LinkedMultiValueMap<String, String> headerParams = new LinkedMultiValueMap<>();
        headers.forEach(headerParams::add);
        return headerParams;
    }

    public static LinkedMultiValueMap<String,String> praseBodyParams(Map<String, Object> bodyParams){
        LinkedMultiValueMap<String, String> linkedMultiValueMap = new LinkedMultiValueMap<>();
        if(MapUtils.isNotEmpty(bodyParams)){
            for (Map.Entry<String, Object> entry : bodyParams.entrySet()) {
                Object value = entry.getValue();
                if (null != value){
                    linkedMultiValueMap.put(entry.getKey(), Lists.newArrayList(value.toString()));
                }
            }
        }
        return linkedMultiValueMap;
    }
}
