/*
 * Copyright 2016 bianxianmao.com All right reserved. This software is the confidential and proprietary information of
 * textile.com ("Confidential Information"). You shall not disclose such Confidential Information and shall use it only
 * in accordance with the terms of the license agreement you entered into with bianxianmao.com.
 */

package com.bxm.warcar.validate.factory;

import com.bxm.warcar.utils.ReflectUtils;
import com.bxm.warcar.validate.Validator;
import com.bxm.warcar.validate.annotation.*;
import com.bxm.warcar.validate.internal.*;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang.reflect.FieldUtils;
import org.apache.commons.lang3.StringUtils;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.Date;
import java.util.List;
import java.util.Map;

public class AnnotationValidatorFactory implements ValidatorFactory {

	private final Map<Class<? extends Annotation>, ValidatorMetadata> validatorMetadatas = Maps
			.newHashMap();

	public AnnotationValidatorFactory() {
		initValidatorMetadatas();
	}

	private void initValidatorMetadatas() {
		registerValidatorMetadata(new ValidatorMetadata(ValidateCollection.class) {

			@Override
			public Validator create(Field f, Annotation anno) {
				ValidateCollection an = (ValidateCollection) anno;
				Class<? extends Validator> clazz = an.elementValidator();
				try {
					return new CollectionValidator(f, clazz.newInstance(), an.allowEmpty(), an
							.minSize(), an.maxSize());
				} catch (InstantiationException e) {
					throw new ValidatorCreateException("CollectionValidator create:", e);
				} catch (IllegalAccessException e) {
					throw new ValidatorCreateException("CollectionValidator create:", e);
				}
			}

			@Deprecated
			@Override
			public Validator create(Field f, Annotation anno, String defaultValue) {
				return create(f, anno);
			}


			@Override
			public void match(Class<?> clazz) {
				if (!(List.class.isAssignableFrom(clazz) || clazz.isArray()))
					throw new ValidateAnnotationMisuseException(getAnnoClass(),
							" expect List or Array,but " + clazz);
			}
		});
		registerValidatorMetadata(new ValidatorMetadata(ValidateLength.class) {

			@Override
			public Validator create(Field f, Annotation anno) {
				ValidateLength an = (ValidateLength) anno;
				return new LengthValidator(f, an.min(), an.max());
			}

			@Override
			public Validator create(Field f, Annotation anno, String defaultValue) {
				ValidateLength an = (ValidateLength) anno;
				return new LengthValidator(f, an.min(), an.max(), defaultValue);
			}

			@Override
			public void match(Class<?> clazz) {
				if (String.class != clazz)
					throw new ValidateAnnotationMisuseException(getAnnoClass(),
							" expect String,but " + clazz);
			}
		});
		registerValidatorMetadata(new ValidatorMetadata(ValidateNotGreatThan.class) {

			@Override
			public Validator create(Field f, Annotation anno) {
				ValidateNotGreatThan an = (ValidateNotGreatThan) anno;
				String cfn = an.compareField();
				Field cf;
				try {
					cf = f.getDeclaringClass().getDeclaredField(cfn);
				} catch (SecurityException e) {
					throw new ValidatorCreateException("NotGreatThanValidator create:", e);
				} catch (NoSuchFieldException e) {
					throw new ValidatorCreateException("NotGreatThanValidator create:", e);
				}
				// TODO: if cf is not Numeric
				return new NotGreatThanValidator(f, cf);
			}

			@Deprecated
			@Override
			public Validator create(Field f, Annotation anno, String defaultValue) {
				return create(f, anno) ;
			}

			@Override
			public void match(Class<?> clazz) {
				if (!(Byte.class == clazz || byte.class == clazz || Short.class == clazz
						|| short.class == clazz || Integer.class == clazz || int.class == clazz
						|| Long.class == clazz || long.class == clazz)) {
					throw new ValidateAnnotationMisuseException(getAnnoClass(),
							" expect Numeric,but " + clazz);
				}
			}
		});
		registerValidatorMetadata(new ValidatorMetadata(ValidateNotLaterThan.class) {

			@Override
			public Validator create(Field f, Annotation anno) {
				ValidateNotLaterThan an = (ValidateNotLaterThan) anno;
				String cfn = an.compareField();
				Field cf;
				try {
					cf = f.getDeclaringClass().getField(cfn);
				} catch (SecurityException e) {
					throw new ValidatorCreateException("NotLaterThanValidator create:", e);
				} catch (NoSuchFieldException e) {
					throw new ValidatorCreateException("NotLaterThanValidator create:", e);
				}
				// TODO: if cf is not Date
				return new NotLaterThanValidator(f, cf);
			}

			@Deprecated
			@Override
			public Validator create(Field f, Annotation anno, String defaultValue) {
				return create(f, anno);
			}

			@Override
			public void match(Class<?> clazz) {
				if (!(Date.class.isAssignableFrom(clazz)))
					throw new ValidateAnnotationMisuseException(getAnnoClass(), " expect Date,but "
							+ clazz);
			}
		});
		registerValidatorMetadata(new ValidatorMetadata(ValidateNotNull.class) {

			@Override
			public Validator create(Field f, Annotation anno) {
				return new NotNullValidator(f);
			}

			@Override
			public Validator create(Field f, Annotation anno, String defaultValue) {
				return new NotNullValidator(f, defaultValue);
			}

			@Override
			public void match(Class<?> clazz) {
				if (clazz.isPrimitive())
					throw new ValidateAnnotationMisuseException(getAnnoClass(),
							" expect Object,but " + clazz);
			}
		});
		registerValidatorMetadata(new ValidatorMetadata(ValidateImeiOrIdfa.class) {

			@Override
			public Validator create(Field f, Annotation anno) {
				return new ImeiOrIdfalValidator(f);
			}

			@Override
			public Validator create(Field f, Annotation anno, String defaultValue) {
				return new ImeiOrIdfalValidator(f, defaultValue);
			}

			@Override
			public void match(Class<?> clazz) {
				if (clazz.isPrimitive())
					throw new ValidateAnnotationMisuseException(getAnnoClass(),
							" expect Object,but " + clazz);
			}
		});
		registerValidatorMetadata(new ValidatorMetadata(ValidateNumeric.class) {

			@Override
			public Validator create(Field f, Annotation anno) {
				ValidateNumeric an = (ValidateNumeric) anno;
				return new NumericValidator(f, an.min(), an.max());
			}

			@Override
			public Validator create(Field f, Annotation anno, String defaultValue) {
				return create(f, anno, defaultValue);
			}

			@Override
			public void match(Class<?> clazz) {
				if (!(Byte.class == clazz || byte.class == clazz || Short.class == clazz
						|| short.class == clazz || Integer.class == clazz || int.class == clazz
						|| Long.class == clazz || long.class == clazz || String.class == clazz)) {
					throw new ValidateAnnotationMisuseException(getAnnoClass(),
							" expect Numeric,but " + clazz);
				}
			}
		});
		registerValidatorMetadata(new ValidatorMetadata(ValidatePattern.class) {

			@Override
			public Validator create(Field f, Annotation anno) {
				ValidatePattern an = (ValidatePattern) anno;
				return new PatternValidator(f, an.value());
			}

			@Override
			public Validator create(Field f, Annotation anno, String defaultValue) {
				ValidatePattern an = (ValidatePattern) anno;
				return new PatternValidator(f, an.value(), defaultValue);
			}

			@Override
			public void match(Class<?> clazz) {
				if (String.class != clazz)
					throw new ValidateAnnotationMisuseException(getAnnoClass(),
							" expect String,but " + clazz);
			}
		});
		registerValidatorMetadata(new ValidatorMetadata(ValidateCustom.class) {

			@Override
			public Validator create(Field f, Annotation anno) {
				ValidateCustom an = (ValidateCustom) anno;
				Class<? extends Validator> clazz = an.validator();
				try {
					return new CustomValidator(f, clazz.newInstance());
				} catch (InstantiationException e) {
					throw new ValidatorCreateException("CustomValidator create:", e);
				} catch (IllegalAccessException e) {
					throw new ValidatorCreateException("CustomValidator create:", e);
				}
			}

			@Deprecated
			@Override
			public Validator create(Field f, Annotation anno, String defaultValue) {
				return create(f, anno);
			}

			@Override
			public void match(Class<?> clazz) {
				// Empty Handler
			}
		});
	}

	@Override
	public Validator create(Class<?> beanClass) {
		final List<Validator> childValidators = createValidator(beanClass);

		if (childValidators.isEmpty())
			return null;
		final BeanValidator beanValidator = new BeanValidator();
		beanValidator.register(childValidators);
		return beanValidator;
	}

//	@Override
//	public Validator create(Class<?> beanClass) {
//		final List<Validator> childValidators = Lists.newArrayList();
//		ReflectUtils.findFields(beanClass, new ReflectUtils.FieldFilter() {
//
//			@Override
//			public boolean accept(Field _f) {
//				List<Validator> vs = create(_f);
//				childValidators.addAll(vs);
//				return false;
//			}
//		});
//
//		if (childValidators.isEmpty())
//			return null;
//		final BeanValidator beanValidator = new BeanValidator();
//		beanValidator.register(childValidators);
//		return beanValidator;
//	}

	private List<Validator> create(Field f) {
		List<Validator> ret = Lists.newArrayList();
		Class<?> clazz = f.getType();

		// annotations
		Annotation[] annos = f.getAnnotations();
		for (Annotation anno : annos) {
			ValidatorMetadata vm = validatorMetadatas.get(anno.annotationType());
			if (null == vm)
				continue;
			vm.match(clazz);
			Validator validator = vm.create(f, anno);
			ret.add(validator);
		}

		// child object
		final List<Validator> childValidators = Lists.newArrayList();
		if (clazz.isPrimitive() || (String.class == clazz))
			return ret;
		ReflectUtils.findFields(clazz, new ReflectUtils.FieldFilter() {

			@Override
			public boolean accept(Field _f) {
				List<Validator> vs = create(_f);
				childValidators.addAll(vs);
				return false;
			}
		});
		if (childValidators.size() > 0) {
			BeanValidator beanValidator = new BeanValidator();
			for (Validator childValidator : childValidators) {
				beanValidator.register(childValidator);
			}

			ret.add(new BeanValidatorWrapper(f, beanValidator));
		}
		return ret;
	}

	private List<Validator> createValidator(Class<?> beanClass) {
		List<Validator> ret = Lists.newArrayList();

		do {
			Field[] declaredFields = beanClass.getDeclaredFields();
			for (Field declaredField : declaredFields) {
				Annotation[] annotations = declaredField.getAnnotations();
				String defaultValue = getDefaultValue(annotations);

				for (Annotation annotation : annotations) {
					ValidatorMetadata validatorMetadata = validatorMetadatas.get(annotation.annotationType());
					if (null == validatorMetadata) {
						continue;
					}
					validatorMetadata.match(declaredField.getType());

					Validator validator;
					if(defaultValue != null) {
						validator = validatorMetadata.create(declaredField, annotation, defaultValue);
					}else{
						validator = validatorMetadata.create(declaredField, annotation);
					}
					ret.add(validator);
				}
			}
		}
		while (null != (beanClass = beanClass.getSuperclass()));
		return ret;
	}

	private String getDefaultValue(Annotation[] annotations) {
		for (Annotation annotation : annotations) {
			if(annotation.annotationType().equals(DefaultValue.class)){
				DefaultValue an = (DefaultValue) annotation;
				return an.value() != null ? an.value() : null;
			}
		}
		return null;
	}

	public void registerValidatorMetadata(ValidatorMetadata metadata) {
		validatorMetadatas.put(metadata.getAnnoClass(), metadata);
	}
}
