package com.bxm.warcar.web.util;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;

import com.bxm.warcar.web.util.serialization.ByteArraySerialization;
import com.bxm.warcar.web.util.serialization.JsonSerialization;
import com.bxm.warcar.web.util.serialization.PrimitiveSerialization;

/**
 * 声明这个方法使用加解密处理，可以对请求体解密，对返回内容加密。
 * 需要与{@link EncryptedMessage}配合使用，其中{@link #model()}返回的数据模型类需要与{@link EncryptedMessage#getObject()}泛类型一致，否则可能会抛出{@link ClassCastException}。
 *
 * @author allen
 * @date 2020-09-30
 * @since 1.0
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.METHOD)
public @interface Encrypted {

    /**
     * 数据模型，从密文里解密后得到的数据进行反序列化时的对象类型。
     * @return 模型
     */
    Class<?> model();

    /**
     * <p>如果当前注解的方法返回的对象是{@link org.springframework.http.ResponseEntity}，那么会创建一个新的对象，复制 {@code HttpStatus} 和 {@code HttpHeaders}，然后对原始 {@code body} 进行序列化后再加密处理并返回。</p>
     * <p>对原始 {@code body} 序列化的方式需要通过 {@link #serialization()} 来指定实现类。</p>
     *
     * @return 是否对返回内容进行加密处理
     * @see #encryptResponseHttpStatus()
     */
    boolean encryptResponseEntity() default false;

    /**
     * <p>当返回内容需要加密处理时，即 {@link #encryptResponseEntity()} 返回为 {@code true}，会再判断返回的 {@code StatusCode} 是否包含在返回的 {@link HttpStatus} 数组中。</p>
     * <p>如果包含，则进行加密，否则不会加密。</p>
     * <p><b>只对方法返回的对象类型为 {@link ResponseEntity} 时有效。</b></p>
     * @return 默认 {@link HttpStatus#OK}
     * @see HttpStatus
     */
    HttpStatus[] encryptResponseHttpStatus() default HttpStatus.OK;

    /**
     * 当发生解密请求数据发生异常时，是否终止处理接下来的业务逻辑，并直接返回 {@link org.springframework.http.HttpStatus#BAD_REQUEST} 状态码。
     * @return 默认终止处理
     */
    boolean interruptForException() default true;

    /**
     * <p>请求参数解密后的文本格式</p>
     * @return 默认 {@link TextFormat#JSON}
     * @see TextFormat
     */
    TextFormat plaintextFormat() default TextFormat.JSON;

    /**
     * <p>指定序列化实现类</p>
     * <p>选择不同序列化实现的参考建议：</p>
     * <p>{@code byte[]} 类型，建议使用 {@link ByteArraySerialization}</p>
     * <p>{@code String、Short、Byte、Integer、Float、Double、Long、Boolean、BigDecimal} 基础类型，建议使用 {@link PrimitiveSerialization}</p>
     * <p>{@link Object} 对象类型，建议使用 {@link JsonSerialization}</p>
     *
     * <p><i>如果不满足你的需求，可以扩展 {@link Serialization} 接口来实现自定义序列化方式，但是必须保证实现类有一个空的构造函数。</i></p>
     *
     * @return 默认返回 {@code JSON} 格式的 {@link JsonSerialization}
     * @see JsonSerialization
     * @see PrimitiveSerialization
     * @see ByteArraySerialization
     */
    Class<? extends Serialization> serialization() default JsonSerialization.class;
}
