package com.bxm.warcar.utils.response;

/**
 * <p> 接口返回结果model工厂类 </p>
 *
 * @Author: JandMin
 * @since: 1.0.0
 * @Date: 2018/7/26
 */
public final class ResponseModelFactory {

    private ResponseModelFactory(){}

    /**
     * 创建返回结果，参数自定义设置
     * @param <T>
     * @return
     */
    public static <T> ResponseModel<T> getResponseModel(){
        return ResponseModel.newInstance();
    }

    /**
     * 成功返回
     * @param data 返回结果对象
     * @param <T>
     * @return
     */
    public static  <T> ResponseModel<T> SUCCESS(T data){
        return  SUCCESS(data, ResponseCode.SUCCESS.code, ResponseCode.SUCCESS.message);
    }

    /**
     * 成功返回
     * @param data 返回结果对象
     * @param <T>
     * @return
     */
    public static <T> ResponseModel<T> SUCCESS(T data, Integer code, String message) {
        return getResponseModel(data,true, code, message);
    }

    /**
     * 失败，无参数
     *
     * @param
     * @return com.bxm.warcar.utils.response.ResponseModel<T>
     * @throws
     * @author kk.xie
     * @date 2018/7/31 15:09
     */
    public static <T> ResponseModel<T> FAILED(){
        return FAILED(ResponseCode.ERROR.code, ResponseCode.ERROR.message);
    }

    /**
     * 请求失败
     * @param <T>
     * @return
     */
    public static <T> ResponseModel<T> FAILED400() {
        return FAILED(ResponseCode.REQUEST_ERROR.code, ResponseCode.REQUEST_ERROR.message);
    }

    /**
     * 登录失败
     * @param <T>
     * @return
     */
    public static <T> ResponseModel<T> FAILED401(){
        return FAILED(ResponseCode.LOGIN_ERROR.code,ResponseCode.LOGIN_ERROR.message);
    }

    /**
     * 请求超时
     * @param <T>
     * @return
     */
    public static <T> ResponseModel<T> FAILED408(){
        return FAILED(ResponseCode.TIMEOUT_ERROR.code,ResponseCode.TIMEOUT_ERROR.message);
    }

    /**
     * 系统出错
     * @param <T>
     * @return
     */
    public static <T> ResponseModel<T> FAILED500() {
        return FAILED(ResponseCode.SYSTEM_ERROR.code,ResponseCode.SYSTEM_ERROR.message);
    }

    /**
     * 返回失败，失败编码自定义
     * @param code 失败编码
     * @param message 失败描述
     * @param <T>
     * @return
     */
    public static <T> ResponseModel<T> FAILED(Integer code, String message) {
        return getResponseModel(null,false, code, message);
    }

    /**
     * 创建返回结果
     * @param data
     * @param success
     * @param code
     * @param message
     * @param <T>
     * @return
     */
    public static <T> ResponseModel<T> getResponseModel(T data,boolean success,Integer code,String message){
        ResponseModel response = getResponseModel();
        response.setSuccess(success);
        response.setError(!success);
        response.setCode(code);
        response.setMessage(message);
        if(null != data){
            response.setData(data);
        }
        return response;
    }


    public static enum ResponseCode {
        /**
         * //成功
         */
        SUCCESS(1,"成功"),
        /**
         * 失败
         */
        ERROR(0, "失败"),
        /**
         * 请求出错
         */
        REQUEST_ERROR(400, "the parameters error"),
        /**
         * 登录出错
         */
        LOGIN_ERROR(401, "login error"),
        /**
         * 超时
         */
        TIMEOUT_ERROR(408, "login error"),
        /**
         * 系统错误
         */
        SYSTEM_ERROR(500, "system error");

        Integer code;
        String message;

        ResponseCode(Integer code,String message){
            this.code = code;
            this.message = message;
        }
    }
}
