package com.bxm.warcar.integration.eventbus;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import java.lang.reflect.Method;
import java.util.EventObject;
import java.util.Map;
import java.util.Set;

/**
 * @author allen
 * @since 1.0.0
 */
public abstract class AbstractEventPark implements EventPark, ApplicationContextAware, InitializingBean {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractEventPark.class);

    protected ApplicationContext applicationContext;
    protected abstract EventBus getEventBus();
    protected abstract String getEventParkName();

    @Override
    public void register(EventListener listener) {
        EventBus eventBus = getEventBus();
        eventBus.register(listener);
        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("Registering listeners to {}: {}", getEventParkName(), listener);
        }

        boolean noSubscribeMethod = true;
        Method[] methods = listener.getClass().getMethods();
        for (Method method : methods) {
            if (method.isAnnotationPresent(Subscribe.class)) {
                noSubscribeMethod = false;
            }
        }
        if (noSubscribeMethod) {
            throw new RuntimeException(listener.getClass() + " must have an @Subscribe method!");
        }
    }

    @Override
    public void unregister(EventListener listener) {
        getEventBus().unregister(listener);
    }

    @Override
    public void post(EventObject eventObject) {
        getEventBus().post(eventObject);

        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("EventBus post event: {}", eventObject);
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        Map<String, EventListener> beansOfType = applicationContext.getBeansOfType(EventListener.class);
        Set<Map.Entry<String, EventListener>> entries = beansOfType.entrySet();
        for (Map.Entry<String, EventListener> entry : entries) {
            this.register(entry.getValue());
        }
    }
}
