package com.bxm.warcar.ip.impl;

import com.bxm.warcar.ip.IP;
import com.bxm.warcar.ip.IpLibrary;
import com.bxm.warcar.utils.LifeCycle;
import com.google.common.base.Preconditions;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * ipplus360.com
 *
 * @author allen
 * @since 1.0.0
 */
public class Ipplus360IpLibrary extends LifeCycle implements IpLibrary {

    private static final Logger LOGGER = LoggerFactory.getLogger(Ipplus360IpLibrary.class);

    private byte[] bytes = null;
    private File file;
    private long base_len = 64;
    private long offset_addr = 0;
    private long offset_owner = 0;
    private byte[] offset_infe;
    private Pattern ip_re = Pattern.compile("^((25[0-5]|2[0-4]\\d|[01]?\\d\\d?)\\.){3}(25[0-5]|2[0-4]\\d|[01]?\\d\\d?)$");
    private ReentrantLock lock = new ReentrantLock();

    public Ipplus360IpLibrary(File file) {
        Preconditions.checkNotNull(file);
        this.file = file;
    }

    @Override
    protected void doInit() {
        refresh();
    }

    @Override
    protected void doDestroy() {

    }

    @Override
    public IP find(String ip) {
        // 1699201024|1699202559|亚洲|CN|330100|中国|浙江省|杭州市||120.208335|30.255611|
        String str = findIp(ip);
        if (StringUtils.isBlank(str)) {
            return null;
        }
        String[] array = StringUtils.splitByWholeSeparatorPreserveAllTokens(str, "|");
        if (array.length >= 11) {
            String province = adjustProvince(array[6]);
            String city = StringUtils.removeEnd(array[7], "市");
            return new IP(array[5],
                    province,
                    city,
                    array[8],
                    null,
                    array[10],
                    array[9],
                    null,
                    null,
                    array[4],
                    0,
                    array[3]);
        }
        return null;
    }

    private static final String[] SPECIAL_PROVINCIES = new String[] {
            "北京市", "天津市", "上海市", "重庆市"
    };

    private String adjustProvince(String province) {
        if (ArrayUtils.contains(SPECIAL_PROVINCIES, province)) {
            return StringUtils.removeEnd(province, "市");
        }
        return StringUtils.removeEnd(province, "省");
    }

    @Override
    public void refresh() {
        long start = System.currentTimeMillis();
        this.load();
        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("Ip library (ipplus360) load finished in {} ms", (System.currentTimeMillis() - start));
        }
    }

    private void load() {
        lock.tryLock();
        try {
            bytes = fileTobyte(file);
            byte[] asse = new byte[bytes.length - 16];
            byte[] asse1 = new byte[8];
            byte[] asse2 = new byte[8];
            System.arraycopy(bytes, 16, asse, 0, bytes.length - 16);
            System.arraycopy(bytes, 0, asse1, 0, 8);
            System.arraycopy(bytes, 8, asse2, 0, 8);
            this.offset_addr = bytesToLong(asse1);
            this.offset_owner = bytesToLong(asse2);
            this.offset_infe = asse;
        } catch (Exception e) {
            throw new RuntimeException("load", e);
        } finally {
            lock.unlock();
        }
    }

    private long ipToLong(String ipAddress) {
        long result = 0;
        String[] ipAddressInArray = ipAddress.split("\\.");
        for (int i = 3; i >= 0; i--) {
            long ip = Long.parseLong(ipAddressInArray[3 - i]);
            result |= ip << (i * 8);
        }
        return result;
    }

    private static String bytetoChar(byte[] bytes, Integer src, Integer dst) {
        byte[] asses = new byte[dst];
        System.arraycopy(bytes, src, asses, 0, dst);
        return new String(asses);
    }

    private static Integer bytetoInt(byte[] bytes, Integer src, Integer dst) {
        byte[] asses = new byte[dst];
        System.arraycopy(bytes, src, asses, 0, dst);
        return byteArrayToInt(asses);
    }

    private static long byteArrayToInt2(byte[] b) {
        long num = 0;
        for (int ix = 3; ix >= 0; --ix) {
            num <<= 8;
            num |= (b[ix] & 0xff);
        }
        return num;
    }

    private static long bytetoInt2(byte[] bytes, Integer src, Integer dst) {
        byte[] asses = new byte[4];
        System.arraycopy(bytes, src, asses, 0, 4);
        return byteArrayToInt2(asses);
    }

    public String findIp(String ip) {
        Matcher m = this.ip_re.matcher(ip);
        Long nip;
        if (m.find()) {
            nip = ipToLong(m.group(0));
        } else {
            return "Error IP";
        }
        ;
        long record_min = 0;
        long record_max = this.offset_addr / this.base_len - 1;
        long record_mid = (record_min + record_max) / 2;
        long mult_re_ba_l;
        while (record_max - record_min >= 0) {

            mult_re_ba_l = record_mid * this.base_len;
            Integer mult_re_ba = (int) mult_re_ba_l;
            Long minip;
            Long maxip;
            minip = bytetoInt2(this.offset_infe, mult_re_ba, 4);
            maxip = bytetoInt2(this.offset_infe, mult_re_ba + 4, 4);
            if (nip < minip) {
                record_max = record_mid - 1;
            } else if ((nip.equals(minip)) | (nip > minip & nip < maxip) | (nip.equals(maxip))) {
                Integer addr_begin = bytetoInt(this.offset_infe, mult_re_ba + 8, 8);
                Integer addr_length = bytetoInt(this.offset_infe, mult_re_ba + 16, 8);
                Integer owner_begin = bytetoInt(this.offset_infe, mult_re_ba + 24, 8);
                Integer owner_length = bytetoInt(this.offset_infe, mult_re_ba + 32, 8);
                String wgs_lon = bytetoChar(this.offset_infe, mult_re_ba + 40, 12).trim();
                String wgs_lat = bytetoChar(this.offset_infe, mult_re_ba + 52, 12).trim();
                String addr_bundle = bytetoChar(this.offset_infe, addr_begin, addr_length).trim();
                String owner = bytetoChar(this.offset_infe, owner_begin, owner_length).trim();
                return minip.toString() + "|" + maxip.toString() + "|" + addr_bundle + "|" + wgs_lon + "|" + wgs_lat + "|" + owner;
            } else if (nip > maxip) {
                record_min = record_mid + 1;
            } else {
                return "ERROR Case";
            }
            ;
            record_mid = (record_min + record_max) / 2;
        }
        return "Not Found.";

    }

    private static byte[] fileTobyte(File file) throws IOException {
        FileInputStream in = null;
        try {
            in = new FileInputStream(file);
            byte[] data = new byte[in.available()];
            in.read(data);
            in.close();
            return data;
        } finally {
            IOUtils.closeQuietly(in);
        }
    }

    private static int byteArrayToInt(byte[] b) {
        return b[0] & 0xFF << 0 |
                (b[1] & 0xFF) << 8 |
                (b[2] & 0xFF) << 16 |
                (b[3] & 0xFF) << 24;
    }

    private static long bytesToLong(byte[] b) {

        return ((((long) b[7]) << 56) | (((long) b[6] & 0xff) << 48) | (((long) b[5] & 0xff) << 40)
                | (((long) b[4] & 0xff) << 32) | (((long) b[3] & 0xff) << 24) | (((long) b[2] & 0xff) << 16)
                | (((long) b[1] & 0xff) << 8) | (((long) b[0] & 0xff)));
    }

    public static void main(String[] args) throws IOException {
        String fileName = "E:\\埃文科技\\IP_trial_2018M11_single_WGS84_dat\\IP_trial_2018M11_single_WGS84.dat";
        byte[] bytes = FileUtils.readFileToByteArray(new File(fileName));
        String fingerprint = DigestUtils.md5Hex(bytes);
        System.out.println(fingerprint);
        Ipplus360IpLibrary library = new Ipplus360IpLibrary(new File(fileName));
        library.init();
        System.out.println(library.find("117.136.79.102"));
        System.out.println(library.find("112.17.87.139"));
        System.out.println(library.find("73.32.192.246"));
        System.out.println(library.find("220.176.34.17"));
        System.out.println(library.find("220.191.32.0"));
    }
}
