package com.bxm.warcar.xcache.fetchers;

import com.bxm.warcar.cache.serialization.JSONSerialization;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.xcache.Fetcher;
import com.bxm.warcar.xcache.Target;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import org.apache.commons.collections.KeyValue;
import org.apache.commons.lang.StringUtils;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

/**
 * 基于本地缓存实现
 *
 * @author allen
 * @date 2019/1/7
 * @since 1.0.0
 */
public class LoadingCacheFetcher implements Fetcher {

    private final Fetcher source;
    private final LoadingCache<String, Optional<Object>> cache;
    private final JSONSerialization serialization;

    public LoadingCacheFetcher(Fetcher source) {
        this(source, 5, TimeUnit.MINUTES);
    }

    public LoadingCacheFetcher(Fetcher source, int duration, TimeUnit timeUnit) {
        Preconditions.checkNotNull(source);
        this.source = source;
        this.cache = CacheBuilder.newBuilder()
                .expireAfterWrite(duration, timeUnit)
                .build(new CacheLoader<String, Optional<Object>>() {
                    @Override
                    public Optional<Object> load(String key) throws Exception {
                        return Optional.empty();
                    }
                });
        this.serialization = new JSONSerialization();
    }

    @Override
    public <T> T fetch(Target<T> target) {
        Preconditions.checkNotNull(target);
        try {
            if (target.isSkipNativeCache()) {
                return source.fetch(target);
            }
            Optional<Object> s = this.cache.get(target.getKeyGenerator().generateKey(), () -> {
                T value = source.fetch(target);
                return Optional.ofNullable(value);
            });
            Object value = s.orElse(null);
            if (null == value) {
                return null;
            }
            return (T) value;
        } catch (ExecutionException e) {
            return null;
        }
    }

    @Override
    public <T> List<T> fetchToList(Target<T> target) {
        Preconditions.checkNotNull(target);
        try {
            if (target.isSkipNativeCache()) {
                return source.fetchToList(target);
            }
            Optional<Object> s = this.cache.get(target.getKeyGenerator().generateKey(), () -> {
                List<T> value = source.fetchToList(target);
                return Optional.ofNullable(value);
            });
            Object value = s.orElse(null);
            if (null == value) {
                return null;
            }
            return (List<T>) value;
        } catch (ExecutionException e) {
            return null;
        }
    }

    @Override
    public <T> T hfetch(Target<T> target) {
        Preconditions.checkNotNull(target);
        Preconditions.checkArgument(StringUtils.isNotBlank(target.getField()));
        try {
            if (target.isSkipNativeCache()) {
                return source.hfetch(target);
            }
            String key = KeyBuilder.build(target.getKeyGenerator().generateKey(), target.getField());
            Optional<Object> s = this.cache.get(key, () -> {
                T value = source.hfetch(target);
                return Optional.ofNullable(value);
            });
            Object value = s.orElse(null);
            if (null == value) {
                return null;
            }
            return (T) value;
        } catch (ExecutionException e) {
            return null;
        }
    }

    @Override
    public <T> List<T> hfetchToList(Target<T> target) {
        Preconditions.checkNotNull(target);
        Preconditions.checkArgument(StringUtils.isNotBlank(target.getField()));
        try {
            if (target.isSkipNativeCache()) {
                return source.hfetchToList(target);
            }
            String key = KeyBuilder.build(target.getKeyGenerator().generateKey(), target.getField());
            Optional<Object> s = this.cache.get(key, () -> {
                List<T> value = source.hfetchToList(target);
                return Optional.ofNullable(value);
            });
            Object value = s.orElse(null);
            if (null == value) {
                return null;
            }
            return (List<T>) value;
        } catch (ExecutionException e) {
            return null;
        }
    }

    @Override
    public <T> Map<String, T> hfetchall(Target<T> target) {
        Preconditions.checkNotNull(target);
        try {
            if (target.isSkipNativeCache()) {
                return source.hfetchall(target);
            }
            Optional<Object> s = this.cache.get(target.getKeyGenerator().generateKey(), () -> {
                Map<String, T> value = source.hfetchall(target);
                return Optional.ofNullable(value);
            });
            Object value = s.orElse(null);
            if (null == value) {
                return null;
            }
            return (Map<String, T>) value;
        } catch (ExecutionException e) {
            return null;
        }
    }

    @Override
    public <T> Set<T> sfetchall(Target<T> target) {
        Preconditions.checkNotNull(target);
        try {
            if (target.isSkipNativeCache()) {
                return source.sfetchall(target);
            }
            Optional<Object> s = this.cache.get(target.getKeyGenerator().generateKey(), () -> {
                Set<T> value = source.sfetchall(target);
                return Optional.ofNullable(value);
            });
            Object value = s.orElse(null);
            if (null == value) {
                return null;
            }
            return (Set<T>) value;
        } catch (ExecutionException e) {
            return null;
        }
    }

    @Override
    public <T> List<T> zfetch(Target<T> target) {
        Preconditions.checkNotNull(target);
        try {
            if (target.isSkipNativeCache()) {
                return source.zfetch(target);
            }
            Optional<Object> s = this.cache.get(target.getKeyGenerator().generateKey(), () -> {
                List<T> value = source.zfetch(target);
                return Optional.ofNullable(value);
            });
            Object value = s.orElse(null);
            if (null == value) {
                return null;
            }
            return (List<T>) value;
        } catch (ExecutionException e) {
            return null;
        }
    }

    @Override
    public <T> List<KeyValue> zfetchall(Target<T> target) {
        Preconditions.checkNotNull(target);
        try {
            if (target.isSkipNativeCache()) {
                return source.zfetchall(target);
            }
            Optional<Object> s = this.cache.get(target.getKeyGenerator().generateKey(), () -> {
                List<KeyValue> value = source.zfetchall(target);
                return Optional.ofNullable(value);
            });
            Object value = s.orElse(null);
            if (null == value) {
                return null;
            }
            return (List<KeyValue>) value;
        } catch (ExecutionException e) {
            return null;
        }
    }

    @Override
    public <T> long distinct(Target<T> target) {
        Preconditions.checkNotNull(target);
        try {
            if (target.isSkipNativeCache()) {
                return source.distinct(target);
            }
            Optional<Object> s = this.cache.get(target.getKeyGenerator().generateKey(), () -> {
                long value = source.distinct(target);
                return Optional.of(value);
            });
            Object value = s.orElse(0);
            return (long) value;
        } catch (ExecutionException e) {
            return 0L;
        }
    }

    @Override
    public <T> T getClientOriginal() {
        return (T) this.cache;
    }
}
