package com.bxm.warcar.utils;

import java.util.Map;

/**
 * Map 工具类。
 *
 * <b>ThreadUnsafe：当多线程同时初始化同一个 {@code map} 的 {@code V} 时，有可能发生同时构造一个 {@code initialize} 对象并放入集合中，造成数据丢失。</b>
 *
 * @author allen
 * @date 2021-04-28
 * @since 1.0
 * @see SafeMapHelper
 */
public final class MapHelper {

    /**
     * 获取 Map 集合指定 {@code k} 的 {@code value}，如果为空，那么将 {@code initialize} 作为初始化值插入到集合中。
     * @param map 集合
     * @param k Key
     * @param initialize 初始化值
     * @param <K> K 泛型
     * @param <V> V 泛型
     * @return 如果 {@code k} 的值为空，那么返回 {@code initialize} 值。否则返回已存在的 {@code value} 对象。
     */
    public static <K, V> V get(Map<K, V> map, K k, V initialize) {
        V value = map.get(k);
        if (null == value) {
            map.put(k, initialize);
            return initialize;
        }
        return value;
    }

    /**
     * 获取 Map 集合指定 {@code k} 的 {@code value}，如果为空，那么将调用 {@code initializingValue} 接口获取到值作为初始化值插入到集合中。
     * @param map 集合
     * @param k Key
     * @param initializingValue 初始化接口
     * @param <K> K 泛型
     * @param <V> V 泛型
     * @return 如果 {@code k} 的值为空，那么返回初始化接口 {@link InitializingValue} 值。否则返回已存在的 {@code value} 对象。
     */
    public static <K, V> V get(Map<K, V> map, K k, InitializingValue<V> initializingValue) {
        V value = map.get(k);
        if (null == value) {
            V bean = initializingValue.initializing();
            map.put(k, bean);
            return bean;
        }
        return value;
    }

    public interface InitializingValue<V> {

        /**
         * 初始化
         * @return 对象
         */
        V initializing();
    }
}
