package com.bxm.warcar.utils.http.pool;

import com.bxm.warcar.utils.http.HttpMethod;
import com.bxm.warcar.utils.http.HttpPool;
import com.bxm.warcar.utils.http.OkHttpUtils;
import com.bxm.warcar.utils.http.properties.HttpProperties;
import com.google.common.collect.Maps;
import okhttp3.ConnectionPool;
import okhttp3.OkHttpClient;
import okhttp3.RequestBody;
import okhttp3.Response;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * <h3>Shenjue.java基本描述</h3>
 * <p> 单个请求 {@link OkHttpUtils}</p>
 *
 * @author hcmony
 * @since V1.0.0, 2019/05/21 09:44
 */
public class OkHttpPool implements HttpPool {

    private OkHttpClient mOkHttpClient;

    public OkHttpPool(HttpProperties httpProperties) {
        mOkHttpClient = new OkHttpClient.Builder().connectTimeout(httpProperties.getConnectTimeout(),
                TimeUnit.MILLISECONDS).readTimeout(httpProperties.getSocketTimeout(),
                TimeUnit.MILLISECONDS).
                connectionPool(new ConnectionPool(httpProperties.getMaxTotal(), httpProperties.getDefaultMaxPerRoute(), TimeUnit.MINUTES)).
                build();
    }

    /**
     * get 请求url后面拼接参数
     *
     * @param url http://api.bianxiao.com?test=1
     * @return
     */
    @Override
    public String get(String url) throws IOException {
        return OkHttpUtils.packagingResult(getRequest(url));
    }

    /**
     * get 请求url后面拼接参数
     *
     * @param url     http://api.bianxiao.com?test=1
     * @param headers heders传参
     * @return
     */
    @Override
    public String get(String url, Map<String, String> headers) throws IOException {
        return OkHttpUtils.packagingResult(request(url, HttpMethod.GET, headers));
    }

    /**
     * get 请求
     *
     * @param url     http://api.bianxiao.com
     * @param params  params传参
     * @param headers heders传参
     * @return
     */
    @Override
    public String get(String url, Map<String, Object> params, Map<String, String> headers) throws IOException {
        return get(OkHttpUtils.appendParams(url, params), headers);
    }

    /**
     * post 请求
     *
     * @param url http://api.bianxiao.com
     * @return
     */
    @Override
    public String post(String url) throws IOException {
        return post(url, Maps.newHashMap());
    }

    /**
     * post 请求
     *
     * @param url    http://api.bianxiao.com
     * @param params params传参
     * @return
     */
    @Override
    public String post(String url, Map<String, Object> params) throws IOException {
        return OkHttpUtils.packagingResult(postRequest(url, OkHttpUtils.buildBody(params)));
    }

    /**
     * post 请求
     *
     * @param url     http://api.bianxiao.com
     * @param params  params传参
     * @param headers heders传参
     * @return
     */
    @Override
    public String post(String url, Map<String, Object> params, Map<String, String> headers) throws IOException {
        return OkHttpUtils.packagingResult(request(url, HttpMethod.POST, OkHttpUtils.buildBody(params), headers));
    }

    /**
     * post 请求 支持@RequestBody 传参请求
     *
     * @param url     http://api.bianxiao.com
     * @param o       实体传参
     * @param headers heders传参
     * @return
     */
    @Override
    public String postRequestBody(String url, Object o, Map<String, String> headers) throws IOException {
        return OkHttpUtils.packagingResult(requestBody(url, HttpMethod.POST, OkHttpUtils.buildRequestBody(o), headers));
    }

    public Response postRequest(String url, RequestBody body) throws IOException {
        return request(url, HttpMethod.POST, body);
    }

    public Response getRequest(String url) throws IOException {
        return request(url, HttpMethod.GET);
    }

    public Response request(String url, HttpMethod method) throws IOException {
        return request(url, method, Maps.newHashMap());
    }

    public Response request(String url, HttpMethod method, Map<String, String> headers) throws IOException {
        return request(url, method, null, headers);
    }

    public Response request(String url, HttpMethod method, RequestBody body) throws IOException {
        return request(url, method, body, null);
    }

    public Response request(String url, HttpMethod method, RequestBody body, Map<String, String> headers) throws IOException {
        return mOkHttpClient.newCall(OkHttpUtils.setHeaders(OkHttpUtils.buildRequest(url, method, body), headers).build()).execute();
    }

    /**
     * @param url
     * @param method
     * @param body
     * @param headers
     * @return
     * @throws IOException
     * @RequestBody类请求方式
     */
    public Response requestBody(String url, HttpMethod method, RequestBody body, Map<String, String> headers) throws IOException {
        return mOkHttpClient.newCall(OkHttpUtils.setHeaders(OkHttpUtils.buildRequest(url, method, body), headers).build()).execute();
    }

}
