package com.bxm.warcar.integration.distributed;

import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.core.MethodIntrospector;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.scheduling.annotation.Schedules;

import java.lang.reflect.Method;
import java.util.Map;
import java.util.Set;

/**
 *
 * <p>this bean is not eligible for getting processed by all BeanPostProcessors (for example: not eligible for auto-proxying)</p>
 *
 * <p>如果这个实例使用了类似 AutowiredAnnotationBeanPostProcessor 之后的特性，那么可能会失效。</p>
 * <p>意味着这个类里面不能使用 @Autowired @PostConstruct 等能力。请参考接口 {@link BeanPostProcessor} 的实现</p>
 *
 * @author allen
 * @date 2020-12-17
 * @since 1.0
 */
public class DistributedLockBeanFactory implements BeanPostProcessor {

    private final static Logger logger = LoggerFactory.getLogger(DistributedLockBeanFactory.class);

    private final Map<Method, Object> mapping = Maps.newHashMap();

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        Class<?> targetClass = AopUtils.getTargetClass(bean);
        Map<Method, Set<Scheduled>> annotatedMethods = MethodIntrospector.selectMethods(targetClass,
                new MethodIntrospector.MetadataLookup<Set<Scheduled>>() {
                    @Override
                    public Set<Scheduled> inspect(Method method) {
                        Set<Scheduled> scheduledMethods = AnnotatedElementUtils.getMergedRepeatableAnnotations(
                                method, Scheduled.class, Schedules.class);
                        return (!scheduledMethods.isEmpty() ? scheduledMethods : null);
                    }
                });
        if (annotatedMethods.isEmpty()) {
            if (logger.isTraceEnabled()) {
                logger.trace("No @Scheduled annotations found on bean class: " + bean.getClass());
            }
        }
        else {
            // Non-empty set of methods
            for (Map.Entry<Method, Set<Scheduled>> entry : annotatedMethods.entrySet()) {
                Method method = entry.getKey();
                mapping.put(method, bean);
            }
            if (logger.isInfoEnabled()) {
                logger.info(annotatedMethods.size() + " @Scheduled methods has been found on bean '" + beanName +
                        "': " + annotatedMethods);
            }
        }
        return bean;
    }

    public Map<Method, Object> getMapping() {
        return mapping;
    }
}
