package com.bxm.warcar.id.redis;

import com.bxm.warcar.id.IdGenerator;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.util.Objects;

/**
 * <p>字母数字格式的 ID 生成器。</p>
 *
 * <p><b>不推荐在大并发流量下使用</b></p>
 *
 * 按照 10 位的长度来设置的话，
 * 前缀 2 位，0-9 a-z A-Z 的组合，共 62^2 个字符，后缀 8 位自增，用 0 来补齐长度。
 * 所以这个方案共有 62^2*99999999=384,399,996,156 个唯一字符串。
 * 前缀使用一个 key 来自增值作为索引，不能超过 3843；
 * 后缀使用一个 key 来自增值作为值，当超过 99999999 时，重置 0，且前缀自增。
 *
 * <p>参考：https://bxmrds.yuque.com/mizhsy/qc669f/obh5z8</p>
 * @author allen
 * @date 2022-06-01
 * @since 1.0
 */
@Slf4j
public class RedisAlphanumericIdGenerator implements IdGenerator {

    private final static String DEFAULT_HASH_KEY = "com.bxm.warcar.alphanumericid";
    private final static int DEFAULT_MAX_LENGTH = 10;

    private final static String DEFAULT_FIELD_INDEX = "index";
    private final static String DEFAULT_FIELD_ID = "id";

    private final JedisPool jedisPool;
    private final String pk;
    private final String hashKey;
    private final int reset;
    private final int maxLength;

    /**
     * @param jedisPool 连接池
     */
    public RedisAlphanumericIdGenerator(JedisPool jedisPool) {
        this(jedisPool, StringUtils.EMPTY);
    }

    /**
     * @param jedisPool 连接池
     * @param pk PrimaryKey 用来防止同一产品线，但使用不同数据源可能造成的重复。
     */
    public RedisAlphanumericIdGenerator(JedisPool jedisPool, String pk) {
        this(jedisPool, pk, DEFAULT_HASH_KEY, DEFAULT_MAX_LENGTH);
    }

    /**
     * @param jedisPool 连接池
     * @param pk PrimaryKey 用来防止同一产品线，但使用不同数据源可能造成的重复。
     * @param hashKey hashKey
     * @param maxLength 唯一ID最大长度
     */
    public RedisAlphanumericIdGenerator(JedisPool jedisPool, String pk, String hashKey, int maxLength) {
        this.jedisPool = jedisPool;
        this.pk = Objects.isNull(pk) ? StringUtils.EMPTY : pk;
        this.hashKey = hashKey;
        this.maxLength = maxLength;
        this.reset = computeResetValue(this.pk, maxLength);

        log.info("The reset value is {}", this.reset);
    }

    private int computeResetValue(String pk, int maxLength) {
        // 计算 reset
        int pklen = pk.length();
        int idLen = maxLength - pklen - Prefix.getMaxLen();
        StringBuilder s = new StringBuilder();
        for (int i = 0; i < idLen; i++) {
            s.append('9');
        }
        return NumberUtils.toInt(s.toString());
    }

    @Override
    public String next() {
        try (Jedis jedis = jedisPool.getResource()) {
            long index = 0;

            String cacheIndex = jedis.hget(hashKey, DEFAULT_FIELD_INDEX);
            if (StringUtils.isBlank(cacheIndex)) {
                jedis.hset(hashKey, DEFAULT_FIELD_INDEX, Objects.toString(index));
            } else {
                index = NumberUtils.toLong(cacheIndex);
            }

            long after = jedis.hincrBy(hashKey, DEFAULT_FIELD_ID + ":" + index, 1);
            if (after > reset) {
                index++;
                jedis.hset(hashKey, DEFAULT_FIELD_INDEX, Objects.toString(index));
                after = jedis.hincrBy(hashKey, DEFAULT_FIELD_ID + ":" + index, 1);
            }

            String prefix = Prefix.VALUES[(int) index];
            int idSize = maxLength - pk.length() - prefix.length();
            return pk + prefix + StringUtils.leftPad(Objects.toString(after), idSize, '0');
        } catch (Exception e) {
            return RandomStringUtils.randomAlphanumeric(maxLength);
        }
    }
}
