package com.bxm.warcar.integration.message;

import com.bxm.warcar.integration.message.annotation.Messaging;
import com.bxm.warcar.mq.Producer;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.InvocationTargetException;

/**
 * @author allen
 * @since 1.0.0
 */
public abstract class AbstractMessageProducer implements MessageProducer {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractMessageProducer.class);

    protected final Producer producer;
    protected final Object properties;

    /**
     * @param producer 消息生产者
     * @param properties 配置对象
     */
    public AbstractMessageProducer(Producer producer, Object properties) {
        this.producer = producer;
        this.properties = properties;
    }

    protected abstract void sendMessage(String topic, Object returning, Object arg, Messaging messaging);

    @Override
    public void send(String topic, Object arg, Object returning, Messaging messaging) {
        if (null == returning) {
            return;
        }
        if (null == messaging) {
            return;
        }

        topic = StringUtils.isBlank(topic) ? getTopic(messaging) : topic;
        if (StringUtils.isNotBlank(topic)) {
            sendMessage(topic, returning, arg, messaging);
        }
    }

    @Override
    public Producer getProducer() {
        return producer;
    }

    @Override
    public Object getProperties() {
        return properties;
    }

    private String getTopic(Messaging annotation) {
        try {
            String topic = annotation.value();
            if (StringUtils.isBlank(topic)) {
                topic = BeanUtils.getProperty(properties, annotation.topicFieldNameOfConfiguration());
            }
            return topic;
        } catch (IllegalAccessException | NoSuchMethodException | InvocationTargetException e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("getTopic: ", e);
            }
            return null;
        }
    }
}
