package com.aliyun.dashvector.validator;

import com.aliyun.dashvector.common.DashVectorException;
import com.aliyun.dashvector.common.ErrorCode;
import com.aliyun.dashvector.models.CollectionMeta;
import com.aliyun.dashvector.models.Doc;
import com.aliyun.dashvector.models.requests.QueryDocGroupByRequest;
import com.aliyun.dashvector.models.requests.QueryDocRequest;
import com.aliyun.dashvector.models.requests.WriteDocRequest;
import com.aliyun.dashvector.proto.DocOpResult;
import com.aliyun.dashvector.utils.Validator;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;

import static com.aliyun.dashvector.utils.Validator.*;

/**
 * @author sanyi
 * @date 2025/5/28 14:13
 */
public abstract class VectorValidator {
    protected final CollectionMeta meta;

    protected VectorValidator(CollectionMeta meta) {
        this.meta = meta;
    }

    protected abstract void validateVectorFromDoc(Doc doc, DocOpResult.DocOp docOp);

    protected abstract void validateVectorFromQuery(QueryDocRequest request);

    protected abstract void validateVectorFromQueryGroupBy(QueryDocGroupByRequest request);

    public void validateWriteDocRequest(WriteDocRequest<?> request, DocOpResult.DocOp docOp) {
        // validate batch size
        Validator.verifyBatchSize(request.getDocs());
        // validate partition name
        Validator.verifyPartitionName(request.getPartition());

        // validate docs
        request.getDocs().forEach(doc -> {
            // 1. validate doc id
            verifyDocId(doc.getId());
            // 2. validate vector
            validateVectorFromDoc(doc, docOp);
            // 3. verify fields
            verifyFieldsValue(doc.getFields(), meta.getFieldsSchema());
        });
    }

    public void validateQueryDocRequest(QueryDocRequest request) {
        // validate partition name
        Validator.verifyPartitionName(request.getPartition());

        boolean queryById = StringUtils.isNotEmpty(request.getId());
        boolean queryByVector = request.getVector() != null
                || request.getVectorQuery() != null
                || MapUtils.isNotEmpty(request.getVectors())
                || MapUtils.isNotEmpty(request.getSparseVector())
                || MapUtils.isNotEmpty(request.getSparseVectors());

        // validate query type
        if (queryByVector && queryById) {
            throw new DashVectorException(ErrorCode.EXIST_VECTOR_ID);
        }

        // validate query output fields
        Validator.verifyOutputFields(request.getOutputFields());
        // validate filter
        verifyFilter(request.getFilter());

        if (queryByVector) {
            // validate query by vector
            validateVectorFromQuery(request);
        } else {
            // validate query by id
            verifyDocId(request.getId());
            verifyTopK(request.getTopk(), request.isIncludeVector());
        }
    }

    public void validateQueryDocGroupByRequest(QueryDocGroupByRequest request) {
        // validate partition name
        Validator.verifyPartitionName(request.getPartition());

        boolean queryById = StringUtils.isNotEmpty(request.getId());
        boolean queryByVector = request.getVector() != null
                || MapUtils.isNotEmpty(request.getSparseVector());

        // validate query type
        if (queryByVector && queryById) {
            throw new DashVectorException(ErrorCode.EXIST_VECTOR_ID);
        }

        // validate group by field
        if (StringUtils.isEmpty(request.getGroupByField())) {
            throw new DashVectorException(ErrorCode.INVALID_GROUP_BY, "GroupByField is empty");
        }

        // validate query output fields
        Validator.verifyOutputFields(request.getOutputFields());

        // Group_by does not support query with multi vector
        // Therefore, verify group_topk and group_count on the outermost layer
        verifyGroupTopK(request.getGroupTopk(), request.isIncludeVector());
        verifyGroupCount(request.getGroupCount(), request.isIncludeVector());

        // validate filter
        verifyFilter(request.getFilter());

        if (queryByVector) {
            // validate query by vector
            validateVectorFromQueryGroupBy(request);
        } else {
            // validate query by id
            verifyDocId(request.getId());
        }
    }
}
