package com.bxm.adsprod.timer.jobs;

import com.alibaba.fastjson.JSONArray;
import com.bxm.adsprod.dal.ticket.AdTicketMapper;
import com.bxm.adsprod.facade.ticket.DirectMaterial;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.adsprod.model.dao.ticket.DirectTicketMaterialDto;
import com.bxm.adsprod.timer.Job;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 广告位素材点击率
 * @author bcc
 */

@Component
public class DirectMaterialCTRJob  implements Job {
    private static final Logger LOGGER = LoggerFactory.getLogger(PositionDataPullJob.class);

    /**
     * 曝光阈值
     */
    private static final int EXPOSURE_THRESHOLD = 100 ;

    /**
     * 默认点击率
     */
    private static final double DEFAULT_CTR = 0.9;

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;

    @Autowired
    @Qualifier("jedisUpdater")
    private Updater updater;

    @Autowired
    private AdTicketMapper adTicketMapper;

    private ThreadPoolExecutor executorService = new ThreadPoolExecutor(Runtime.getRuntime().availableProcessors()*2,
            Runtime.getRuntime().availableProcessors()*4,0L, TimeUnit.SECONDS,new LinkedBlockingDeque(),
            new ThreadFactoryBuilder().setNameFormat("DirectMaterialCTRJob-pool-%d").build()
    );

    @Override
    //@Scheduled(cron="0 0/5 * * * ?")
    public void execute() {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Starting  DirectMaterialCTRJob...");
        }
        List<DirectTicketMaterialDto> directTicketMaterials = adTicketMapper.findAllDirectTicketMaterial();

        //券ID尺寸对应的素材ID列表
        Map<String,List<Long>> ticketIdSizeMaterialMap = extractTicketIdSizeAndMaterialIdMap(directTicketMaterials);
        //素材ID对应的广告ID
        Map<Long,Long> materailIdAndTicketId = extractMaterailIdAndTicketIdMap(directTicketMaterials);

        //获取所有广告位对应的素材信息
        final Map<String, String> redisMap = fetcher.hfetchallWithSelector(TicketKeyGenerator.getDirectPositionMaterial(), String.class,0);

        //转换类型
        final Map<String,List<DirectMaterial>> materialMap = transfer2List(redisMap);

        long start = System.currentTimeMillis();

        for(Map.Entry entry:materialMap.entrySet()){
            String positionId = entry.getKey().toString();
            executorService.execute(new DirectMaterialCTRCounter(positionId,ticketIdSizeMaterialMap,materailIdAndTicketId,
                    (List<DirectMaterial>)entry.getValue()));
        }

        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("DirectMaterialCTRJob finished in {} ms!", (System.currentTimeMillis() - start));
        }
    }



    /**
     *
     * @param directTicketMaterials
     * @return
     */
    private Map<Long,Long> extractMaterailIdAndTicketIdMap(List<DirectTicketMaterialDto> directTicketMaterials) {
        Map<Long,Long> map =  new HashMap<>();
        for(DirectTicketMaterialDto dto:directTicketMaterials){
            map.put(dto.getMaterialId(),dto.getTicketId());
        }
        return map;
    }

    /**
     * 数据类型转换
     * @param directTicketMaterials
     * @return  Map<String,List<Long>>   券id和尺寸：素材ID列表
     */
    private Map<String,List<Long>> extractTicketIdSizeAndMaterialIdMap(List<DirectTicketMaterialDto> directTicketMaterials) {
        Map<String,List<Long>> map = new HashMap<>();
        for(DirectTicketMaterialDto dto:directTicketMaterials){
            String key = dto.getTicketId()+"_"+dto.getSize();
            List<Long> materials = map.get(key);
            if(null == materials){
                materials = Lists.newArrayList(dto.getMaterialId());
            }else{
                materials.add(dto.getMaterialId());
            }

            map.put(key,materials);
        }
        return map;
    }

    private Map<String,List<DirectMaterial>> transfer2List(Map<String,String> redisMap) {
        final Map<String,List<DirectMaterial>> materialMap = Maps.newHashMap();
        if(null != redisMap && redisMap.size()>0){
            for(Map.Entry entry:redisMap.entrySet()){
                List<DirectMaterial> dms = JSONArray.parseArray(entry.getValue().toString(),DirectMaterial.class);
                materialMap.put(entry.getKey().toString(),dms);
            }
        }
        return materialMap;
    }

    private KeyGenerator getAssetsKey(int hour,String positionId){
        if(hour == 2){
            return TicketKeyGenerator.Statistics.getTicketAssets2Hour(positionId);
        }else if(hour == 24){
            return TicketKeyGenerator.Statistics.getTicketAssets24Hour(positionId);
        }
        return null;
    }
    /**
     * 获取之前或者之后几个小时的时间
     * @param hour
     * @return
     */
    private static String getTimeByHour(int hour) {

        Calendar calendar = Calendar.getInstance();

        calendar.set(Calendar.HOUR_OF_DAY, calendar.get(Calendar.HOUR_OF_DAY) + hour);

        return new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'").format(calendar.getTime());

    }

    class DirectMaterialCTRCounter implements Runnable{

        private String positionId;

        private Map<String,List<Long>> ticketIdSizeMaterialMap;

        private Map<Long,Long> materailIdAndTicketId ;

        private List<DirectMaterial> directMaterials;

        public DirectMaterialCTRCounter(String positionId,
                                        Map<String,List<Long>> ticketIdSizeMaterialMap,
                                        Map<Long,Long> materailIdAndTicketId,
                                        List<DirectMaterial> directMaterials) {
            this.positionId = positionId;
            this.ticketIdSizeMaterialMap = ticketIdSizeMaterialMap;
            this.materailIdAndTicketId = materailIdAndTicketId;
            this.directMaterials = directMaterials;
        }

        @Override
        public void run() {
            /**
             * 统计每个广告位下，
             * 各个券的素材CTR最高数据
             */
            if(null == directMaterials){

                Set<String> ticketIdAndSizes = fetcher.hfetch(TicketKeyGenerator.getDirectPositionTicket(), positionId, Set.class);
                if(CollectionUtils.isNotEmpty(ticketIdAndSizes)){
                    for(String ticketIdAndSize:ticketIdAndSizes) {
                        updater.hupdate(TicketKeyGenerator.getDirectPositionMaterialCTR(positionId), ticketIdAndSize.split("_")[0], Lists.newArrayList());
                    }
                }
                return;
            }

            //广告ID 对应  最高CTR值
            Map<Long,Double> maxScore = Maps.newHashMap();

            //素材ID 对应 CTR值
            Map<Long,Double> assetAndScoreMap = Maps.newHashMap();

            //素材ID 对应 素材信息
            Map<Long,DirectMaterial> assetAndMaterialMap = Maps.newHashMap();
            for(DirectMaterial dm:directMaterials) {
                double ctr = getCtrByPositionIdAndAssetId(positionId, dm.getId());

                Long  ticketId = materailIdAndTicketId.get(dm.getId());
                if(null != ticketId && ticketId>0){
                    Double maxCtr = maxScore.get(ticketId);
                    if(null == maxCtr || maxCtr <= ctr){
                        maxScore.put(ticketId,ctr);
                        assetAndScoreMap.put(dm.getId(),ctr);
                        assetAndMaterialMap.put(dm.getId(),dm);
                    }
                }
            }
            Set<String> ticketIdAndSizes = fetcher.hfetch(TicketKeyGenerator.getDirectPositionTicket(), positionId, Set.class);
            //计算每个广告位下，各广告券，最高CTR的素材
            if(CollectionUtils.isNotEmpty(ticketIdAndSizes)){
                for(String ticketIdAndSize:ticketIdAndSizes){
                    Double maxCrt = maxScore.get(Long.valueOf(ticketIdAndSize.split("_")[0]));
                    List<Long> materialIds = ticketIdSizeMaterialMap.get(ticketIdAndSize);
                    List<DirectMaterial> maxCtrList = Lists.newArrayList();
                    //广告位不含素材
                    if(maxCrt == null || CollectionUtils.isEmpty(materialIds)){
                        updater.hupdate(TicketKeyGenerator.getDirectPositionMaterialCTR(positionId),ticketIdAndSize.split("_")[0],maxCtrList);
                    }else {
                        for (Map.Entry assetIdAndCtr : assetAndScoreMap.entrySet()) {
                            Long materialId = Long.valueOf(assetIdAndCtr.getKey().toString());
                            Double ctr = Double.valueOf(assetIdAndCtr.getValue().toString());
                            //减0.00001防止浮点计算不准确
                            if (ctr >= maxCrt - 0.00001 && materialIds.contains(materialId)) {
                                maxCtrList.add(assetAndMaterialMap.get(materialId));
                            }
                        }
                        updater.hupdate(TicketKeyGenerator.getDirectPositionMaterialCTR(positionId),ticketIdAndSize.split("_")[0],maxCtrList);
                    }
                }
            }
        }

        private double getCtrByPositionIdAndAssetId(String positionId, Long id) {
            Double ctr2Hour = getCtrValue(id, 2,positionId);
            if (ctr2Hour != null){
                return ctr2Hour;
            }
            Double ctr24Hour = getCtrValue(id, 24,positionId);
            if (ctr24Hour != null) {
                return ctr24Hour;
            }

            //历史曝光
            Long hisView = fetcher.hfetchWithSelector(TicketKeyGenerator.getDirectMaterialView(positionId), id + "", Long.class, 3);
            if(null != hisView && hisView >= EXPOSURE_THRESHOLD){
                Long hisClick = fetcher.hfetchWithSelector(TicketKeyGenerator.getDirectMaterialClick(positionId), id + "", Long.class, 3);
                if(null == hisClick){
                    hisClick = 0L;
                }
                return  hisClick.doubleValue()/hisView.doubleValue();
            }
            return DEFAULT_CTR;
        }

        private Double getCtrValue(Long id, int hour ,String positionId) {
            KeyGenerator assetsKey = getAssetsKey(hour, positionId);
            String viewAndClick = fetcher.hfetch(assetsKey, id+"", String.class);
            if(StringUtils.isNotBlank(viewAndClick)&& viewAndClick.contains("-")){
                String[] viewClick = viewAndClick.split("-");
                Double view = Double.valueOf(viewClick[0].toString());
                if(view>= EXPOSURE_THRESHOLD){
                    return  Double.valueOf(viewClick[1].toString())/Double.valueOf(viewClick[0].toString());
                }
            }
            return null;
        }
    }
}
