package com.bxm.adsprod.timer.jobs;

import com.bxm.adsprod.dal.ticket.AdTicketMapper;
import com.bxm.adsprod.facade.advertiser.AdvertiserService;
import com.bxm.adsprod.facade.commons.CachePushableFields;
import com.bxm.adsprod.facade.ticket.*;
import com.bxm.adsprod.model.dao.ticket.Ticket;
import com.bxm.adsprod.model.so.rules.TicketTimelineRuleSo;
import com.bxm.adsprod.timer.Job;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.message.Message;
import com.bxm.warcar.message.MessageSender;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.math.BigInteger;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * @author allen
 * @since 1.0.0
 */
@Component
public class TicketStatusRefreshJob implements Job {

    private static final Logger LOGGER = LoggerFactory.getLogger(TicketStatusRefreshJob.class);

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;
    @Autowired
    private TicketService ticketService;
    @Autowired
    private AdvertiserService advertiserService;
    @Autowired
    private TicketStatisticsService ticketStatisticsService;
    @Autowired
    private AdTicketMapper adTicketMapper ;

    @Autowired
    private MessageSender messageSender;

    @Override
    //@Scheduled(cron = "0 0 * * * ?")
    public void execute() {
        long start = System.currentTimeMillis();

        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("Starting refresh status...");
        }

        List<Ticket> tickets = adTicketMapper.findAll(new Ticket());
        if (CollectionUtils.isNotEmpty(tickets)) {

            List<Ticket> refreshIfNeeded = Lists.newArrayList();

            for (Ticket ticket : tickets) {
                doFilter(ticket,refreshIfNeeded);
            }

            for (Ticket ticket : refreshIfNeeded) {
                try {
                    if (!ticketService.updateTicketStatus(ticket.getId(), ticket.getStatus(), ticket.getPauseReason())) {
                        if (LOGGER.isWarnEnabled()) {
                            LOGGER.warn("[{}] Updating status to {}-{} return fail", ticket.getId(), ticket.getStatus(), ticket.getPauseReason());
                        }
                    } else {
                        if (LOGGER.isInfoEnabled()) {
                            LOGGER.info("[{}] Updating status to {}-{} success", ticket.getId(), ticket.getStatus(), ticket.getPauseReason());
                        }
                    }
                } catch (Exception e) {
                    if (LOGGER.isErrorEnabled()) {
                        LOGGER.error("", e);
                    }
                    messageSender.send2(new Message("adsprod-timer更新券状态时失败,ticketId is"+ ticket.getId()));
                }
            }
        }

        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("Finished in {} ms", (System.currentTimeMillis() - start));
        }
    }

    private void doFilter(Ticket ticket,List<Ticket> refreshIfNeeded){

        try {
            BigInteger ticketId = ticket.getId();
            Byte status = ticket.getStatus();
            Integer pauseReason = defaultIfEmpty(ticket.getPauseReason(), Ticket.PAUSE_REASON_NONE);
            Date startDate = ticket.getValidStartDate();
            Date endDate = ticket.getValidEndDate();
            if (status == Ticket.STATUS_DELETED) {
                return;
            }

            // 结束了
            boolean isOpenOrPause = status == Ticket.STATUS_OPEN || status == Ticket.STATUS_PAUSE;
            if (isOpenOrPause && pauseReason != Ticket.PAUSE_REASON_ENDED && isAfter(endDate)) {
                ticket.setStatus(Ticket.STATUS_PAUSE);
                ticket.setPauseReason(Ticket.PAUSE_REASON_ENDED);
                refreshIfNeeded.add(ticket);
                return;
            }

            // 从未开始到开始
            if (status == Ticket.STATUS_PAUSE && pauseReason == Ticket.PAUSE_REASON_NO_START
                    && isEqual(startDate)) {
                // 还要判断余额是否足够
                setOpenIfEnought(refreshIfNeeded, ticket);
            }

            // 日预算不足 在0点执行的时候需要重新开启
            if (status == Ticket.STATUS_PAUSE && pauseReason == Ticket.PAUSE_REASON_OUT_OF_DAILYBUDGET
                    && isMidnight()) {
                setOpenIfEnought(refreshIfNeeded, ticket);
            }

            // 时段预算开关
            TicketTimelineRuleSo timelineConf = getTimelineConf(ticketId);
            if (null != timelineConf) {
                boolean closeIfNeeded = isCloseIfNeeded(ticketId, timelineConf.getEntries());
                if (status == Ticket.STATUS_OPEN && closeIfNeeded) {
                    // 时段预算不足
                    ticket.setStatus(Ticket.STATUS_PAUSE);
                    ticket.setPauseReason(Ticket.PAUSE_REASON_OUT_OF_TIMEBUDGET);
                    refreshIfNeeded.add(ticket);
                } else if (status == Ticket.STATUS_PAUSE && pauseReason == Ticket.PAUSE_REASON_OUT_OF_TIMEBUDGET
                        && !closeIfNeeded) {
                    // 时段预算不足需要重新打开
                    ticket.setStatus(Ticket.STATUS_OPEN);
                    ticket.setPauseReason(Ticket.PAUSE_REASON_NONE);
                    refreshIfNeeded.add(ticket);
                }
            }

            //该票券类型为：券码类
            if(Ticket.TYPE_COUPON == ticket.getType()) {
                //票券码基本信息
                Map<String, Object> parameters = Maps.newHashMap();
                parameters.put(CachePushableFields.TICKET_ID, ticketId);
                TicketCouponsInfo ticketCouponsInfo = fetcher.fetch(TicketKeyGenerator.getCouponsInfo(parameters), null, TicketCouponsInfo.class);
                LOGGER.info("ticketId:{}", ticketId);
                if(ticketCouponsInfo == null || ticketCouponsInfo.getUseType() == null){
                    LOGGER.info("ticketCouponsInfo为空,key为：{}", TicketKeyGenerator.getCouponsInfo(parameters).generateKey());
                    return;
                }
                LOGGER.info("ticketCouponsInfo:{}", ticketCouponsInfo.toString());

                Boolean disable = false;
                if (TicketCouponsInfo.AdCouponsUseType.USETYPE_SINGLE.getType() == ticketCouponsInfo.getUseType()
                        || TicketCouponsInfo.AdCouponsUseType.USETYPE_MULTIPLE.getType() == ticketCouponsInfo.getUseType()) {
                    //判断未使用的数量如果等于0，则暂停
                    if(getCouponUnUsedCount(TicketKeyGenerator.Coupon.getCouponInfo(ticketId,
                            TicketCouponsCode.AdCouponsStatus.STATUS_UNUSED.getStatus())) <= 0){
                        disable = true;
                    }
                } else if (TicketCouponsInfo.AdCouponsUseType.USETYPE_LIMITED_TIME.getType() == ticketCouponsInfo.getUseType()) {
                    //判断未使用的时间
                    long currentTime = new Date().getTime();
                    if(!(currentTime >= ticketCouponsInfo.getCodeStartTime().getTime() && currentTime < ticketCouponsInfo.getCodeEndTime().getTime())){
                        disable = true;
                    }
                } else if (TicketCouponsInfo.AdCouponsUseType.USETYPE_LIMITED_NUM.getType() == ticketCouponsInfo.getUseType()) {
                    Long receivedCount = getCouponStatusCount(TicketKeyGenerator.Coupon.getCouponInfo(ticketId,
                            TicketCouponsCode.AdCouponsStatus.STATUS_RECEIVED.getStatus()));
                    Long usedCount = getCouponStatusCount(TicketKeyGenerator.Coupon.getCouponInfo(ticketId,
                            TicketCouponsCode.AdCouponsStatus.STATUS_USED.getStatus()));
                    Integer couponUnUsedCount = ticketCouponsInfo.getCodeLimitNum() - (receivedCount.intValue() + usedCount.intValue());
                    LOGGER.info("receivedCount:{}, usedCount:{}, couponUnUsedCount:{}, ",receivedCount, usedCount, couponUnUsedCount);
                    if(couponUnUsedCount <= 0){
                        disable = true;
                        LOGGER.info("ticketId:{} 没有库存，禁用",ticketId);
                    }
                }
                if(disable){
                    // 码库存不足
                    ticket.setStatus(Ticket.STATUS_PAUSE);
                    ticket.setPauseReason(Ticket.PAUSE_REASON_UNDER_STOCK);//码库存不足
                    refreshIfNeeded.add(ticket);
                }
            }

        }catch (Exception e){
            LOGGER.error("", e);
            messageSender.send2(new Message("adsprod-timer更新券状态时失败,ticketId is"+ ticket.getId()));
        }
    }

    private Long getCouponUnUsedCount(KeyGenerator key){
        Object original = fetcher.getClientOriginal();
        if (original instanceof JedisPool) {
            JedisPool jedisPool = (JedisPool) original;

            Jedis jedis = jedisPool.getResource();

            try {
                LOGGER.info("redisKey:{},", key.generateKey());
                return jedis.scard(key.generateKey());
            } finally {
                if (null != jedis) {
                    jedis.close();
                }
            }
        }
        return null;
    }

    private Long getCouponStatusCount(KeyGenerator key){
        Object original = fetcher.getClientOriginal();
        if (original instanceof JedisPool) {
            JedisPool jedisPool = (JedisPool) original;

            Jedis jedis = jedisPool.getResource();

            try {
                LOGGER.info("redisKey:{},", key.generateKey());
                return jedis.hlen(key.generateKey());
            } finally {
                if (null != jedis) {
                    jedis.close();
                }
            }
        }
        return null;
    }

    private int defaultIfEmpty(Integer v, int defaultValue) {
        return null == v ? defaultValue : v;
    }

    private void setOpenIfEnought(List<Ticket> refreshIfNeeded, Ticket ticket) {
        boolean isCPA = !ticket.isCPC();
        if (isEnought(ticket.getAdvertiser()) || isCPA) {
            Integer budgetDaily = ticket.getBudgetDaily();
            if (null != budgetDaily && budgetDaily > 0) {
                ticket.setStatus(Ticket.STATUS_OPEN);
                ticket.setPauseReason(Ticket.PAUSE_REASON_NONE);
                refreshIfNeeded.add(ticket);
            } else {
                ticket.setStatus(Ticket.STATUS_PAUSE);
                ticket.setPauseReason(Ticket.PAUSE_REASON_OUT_OF_DAILYBUDGET);
                refreshIfNeeded.add(ticket);
            }
        } else {
            ticket.setStatus(Ticket.STATUS_PAUSE);
            ticket.setPauseReason(Ticket.PAUSE_REASON_BALANCE_NOT_ENOUGHT);
            refreshIfNeeded.add(ticket);
        }
    }

    private boolean isEnought(BigInteger advertiser) {
        Long balance = advertiserService.getAdvertiserBalance(advertiser);
        return (null != balance && balance > 0);
    }

    private boolean isCloseIfNeeded(BigInteger ticketId, List<TicketTimelineRuleSo.Entry> entries) {
        int currentHour = Calendar.getInstance().get(Calendar.HOUR_OF_DAY);
        for (TicketTimelineRuleSo.Entry entry : entries) {
            int startHour = entry.getStartHour();
            int endHour = entry.getEndHour();
            boolean isNow = currentHour >= startHour && currentHour < endHour;
            if (isNow) {
                long currentBudget = ticketStatisticsService.getBudgetOfTimeline(ticketId, startHour, endHour);
                return (currentBudget >= entry.getLimit());
            }
        }
        return true;
    }

    private TicketTimelineRuleSo getTimelineConf(BigInteger ticketId) {
        return fetcher.fetch(TicketKeyGenerator.Filter.getTimeline(ticketId), null, TicketTimelineRuleSo.class);
    }

    private static boolean isMidnight() {
        return LocalDateTime.now().getHour() == 0;
    }

    private static boolean isAfter(Date date) {
        return LocalDate.now().isAfter(toLocalDate(date));
    }

    private static boolean isEqual(Date date) {
        return LocalDate.now().isEqual(toLocalDate(date));
    }

    private static LocalDate toLocalDate(Date date) {
        return date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
    }
}
