package com.bxm.dailyegg.task.cache;

import cn.hutool.core.date.SystemClock;
import com.bxm.dailyegg.common.enums.TaskActionEnum;
import com.bxm.dailyegg.task.config.TaskProperties;
import com.bxm.dailyegg.task.constant.RedisCacheConstant;
import com.bxm.dailyegg.task.model.bo.UserTaskItemBO;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.google.common.collect.Maps;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.Map;
import java.util.Objects;

import static cn.hutool.core.date.DatePattern.PURE_DATE_FORMAT;
import static com.bxm.dailyegg.task.constant.RedisCacheConstant.TODAY_USER_TASK_CACHE;
import static com.bxm.newidea.component.tools.DateUtils.PATTERN_NO_DELIMITER_FORMAT;

@Component
@AllArgsConstructor
@Slf4j
public class TaskCacheHolder {

    private RedisSetAdapter redisSetAdapter;

    private RedisStringAdapter redisStringAdapter;

    private RedisHashMapAdapter redisHashMapAdapter;

    private TaskProperties taskProperties;

    /**
     * 添加用户获得新人引导奖励记录
     *
     * @param userId 用户id
     */
    public void addGuideAwardRecord(Long userId) {
        redisSetAdapter.add(buildKey(userId), userId);
    }

    /**
     * 用户是否获取过新人引导奖励
     *
     * @param userId 用户id
     * @return boolean
     */
    public boolean hasGuideAwardRecord(Long userId) {
        return redisSetAdapter.exists(buildKey(userId), userId);
    }

    private KeyGenerator buildKey(Long userId) {
        return RedisCacheConstant.USER_GUIDE_AWARD_CACHE.copy().appendKey(userId % 100);
    }

    private KeyGenerator buildBubbleTaskKey(Long userId) {
        return RedisCacheConstant.USER_BUBBLE_TASK_CACHE.copy().appendKey(
            PATTERN_NO_DELIMITER_FORMAT.get().format(new Date())
        ).appendKey(userId);
    }

    /**
     * 用户今日已经完成的气泡任务数量
     *
     * @param userId 用户id
     * @return {@link Integer}
     */
    public Integer getTodayBubbleTaskNum(Long userId) {
        return redisStringAdapter.getInt(buildBubbleTaskKey(userId));
    }

    /**
     * 增加用户今日完成的气泡任务的数量
     *
     * @param userId 用户ID
     */
    public void addUserBubbleTask(Long userId) {
        KeyGenerator key = buildBubbleTaskKey(userId);
        redisStringAdapter.increment(key);
        redisStringAdapter.expireWithDisrupt(key, DateUtils.getCurSeconds());
    }

    /**
     * 2.0.0 获取今日气泡完成情况的数量
     *
     * @param userId 用户ID
     * @return 两种气泡的今日完成次数
     */
    public Map<String, Integer> getTodayBubbleV3(Long userId) {
        KeyGenerator key = buildBubbleV3Key(userId);
        return redisHashMapAdapter.entries(key, Integer.class);
    }

    /**
     * 增加今日气泡完成次数
     *
     * @param userId 用户ID
     * @param action 气泡类型
     */
    public void addTodayBubbleV3Times(Long userId, TaskActionEnum action) {
        KeyGenerator key = buildBubbleV3Key(userId);
        redisHashMapAdapter.increment(key, action.name(), 1);
        redisHashMapAdapter.expireWithDisrupt(key, DateUtils.getCurSeconds());
    }

    private KeyGenerator buildBubbleV3Key(Long userId) {
        return RedisCacheConstant.BUBBLE_TASK_V3_CACHE.copy()
                                                      .appendKey(PURE_DATE_FORMAT.format(SystemClock.now()))
                                                      .appendKey(userId);
    }

    public Map<String, UserTaskItemBO> initTaskCache(Long userId) {
        Map<String, UserTaskItemBO> cacheData = Maps.newHashMap();
        // 初始化今日缓存
        Map<String, TaskProperties.TaskConfigV3> taskConfigV3Map = taskProperties.getTaskConfigV3Map();

        for (Map.Entry<String, TaskProperties.TaskConfigV3> configV3Entry : taskConfigV3Map.entrySet()) {
            TaskProperties.TaskConfigV3 value = configV3Entry.getValue();

            UserTaskItemBO item = new UserTaskItemBO();
            item.setTaskId(-1L);
            item.setCurrent(0);
            item.setManual(value.getManual());
            item.setMax(value.getMax());
            item.setOrder(value.getOrder());
            item.setTaskAction(configV3Entry.getKey());
            item.setFinish(false);

            cacheData.put(configV3Entry.getKey(), item);
        }

        KeyGenerator key = buildTodayTaskCacheKey(userId);
        redisHashMapAdapter.putAll(key, cacheData);
        redisHashMapAdapter.expireWithDisrupt(key, DateUtils.getCurSeconds());

        return cacheData;
    }

    public Map<String, UserTaskItemBO> fetchUserTask(Long userId) {
        KeyGenerator key = buildTodayTaskCacheKey(userId);
        return redisHashMapAdapter.entries(key, UserTaskItemBO.class);
    }

    public void addTaskTimes(Long userId, TaskActionEnum action) {
        if (log.isDebugEnabled()) {
            log.debug("用户[{}]增加任务[{}]次数", userId, action);
        }

        KeyGenerator key = buildTodayTaskCacheKey(userId);
        UserTaskItemBO cacheItem = redisHashMapAdapter.get(key, action.name(), UserTaskItemBO.class);

        if (null == cacheItem) {
            cacheItem = initTaskCache(userId).get(action.name());
            if (null == cacheItem) {
                log.error("不存在对应的任务，用户ID：{},任务：{}", userId, action);
                return;
            }
        }

        if (log.isDebugEnabled()) {
            if (cacheItem.getCurrent() >= cacheItem.getMax()) {
                log.debug("用户[{}]完成任务[{}]次数达标", userId, action);
            }
        }

        if (!cacheItem.getFinish() && cacheItem.getCurrent() < cacheItem.getMax()) {
            cacheItem.setCurrent(cacheItem.getCurrent() + 1);

            if (!cacheItem.getManual() && Objects.equals(cacheItem.getCurrent(), cacheItem.getMax())) {
                cacheItem.setFinish(true);
            }

            redisHashMapAdapter.put(key, action.name(), cacheItem);
        }
    }

    public void setTaskFinish(Long userId, TaskActionEnum action) {
        KeyGenerator key = buildTodayTaskCacheKey(userId);
        UserTaskItemBO cacheItem = redisHashMapAdapter.get(key, action.name(), UserTaskItemBO.class);
        if (!cacheItem.getFinish()) {
            cacheItem.setFinish(true);
            redisHashMapAdapter.put(key, action.name(), cacheItem);
        }
    }

    private KeyGenerator buildTodayTaskCacheKey(Long userId) {
        return TODAY_USER_TASK_CACHE.copy()
                                    .appendKey(PURE_DATE_FORMAT.format(SystemClock.now()))
                                    .appendKey(userId);
    }
}
