package com.bxm.dailyegg.task.service.impl;

import com.bxm.dailyegg.common.enums.TaskActionEnum;
import com.bxm.dailyegg.common.model.param.BaseUserParam;
import com.bxm.dailyegg.task.cache.TaskCacheHolder;
import com.bxm.dailyegg.task.config.TaskProperties;
import com.bxm.dailyegg.task.constant.LogicConstant;
import com.bxm.dailyegg.task.facade.TaskConfigFacadeService;
import com.bxm.dailyegg.task.model.bo.UserTaskItemBO;
import com.bxm.dailyegg.task.model.dto.BubbleTaskItemV3DTO;
import com.bxm.dailyegg.task.model.dto.TaskListItemV2DTO;
import com.bxm.dailyegg.task.service.TaskV3Service;
import com.bxm.dailyegg.task.service.context.UserTaskContext;
import com.bxm.dailyegg.user.dto.UserAccountDTO;
import com.bxm.dailyegg.user.facade.UserAccountFacadeService;
import com.bxm.newidea.component.filter.LogicFilterChainExecutor;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@AllArgsConstructor
@Slf4j
@Service
public class TaskV3ServiceImpl implements TaskV3Service {

    private TaskCacheHolder taskCacheHolder;

    private TaskProperties taskProperties;

    private UserAccountFacadeService userAccountFacadeService;

    private TaskConfigFacadeService taskConfigFacadeService;

    private LogicFilterChainExecutor logicFilterChainExecutor;

    @Override
    public List<BubbleTaskItemV3DTO> fetchBubbleTaskListV3(BaseUserParam param) {
        // 每次返回4个气泡,2个直接获取,2个视频气泡
        Map<String, Integer> todayBubbleV3Map = taskCacheHolder.getTodayBubbleV3(param.getUserId());

        int freeBubble = taskProperties.getBubbleTaskShowNum() / 2;
        int todayFreeBubbleNum = todayBubbleV3Map.getOrDefault(TaskActionEnum.BUBBLE_VIDEO_DIRECT_V3.name(), 0);
        int lastFreeBubbleNum = taskProperties.getMaxFreeBubble() - todayFreeBubbleNum;

        freeBubble = Math.min(freeBubble, lastFreeBubbleNum);

        int videoBubble = taskProperties.getBubbleTaskShowNum() - freeBubble;
        int todayVideoBubbleNum = todayBubbleV3Map.getOrDefault(TaskActionEnum.BUBBLE_VIDEO_V3.name(), 0);
        int lastVideoBubbleNum = taskProperties.getMaxVideoBubble() - todayVideoBubbleNum;
        videoBubble = Math.min(videoBubble, lastVideoBubbleNum);

        UserAccountDTO userAccountDTO = userAccountFacadeService.loadUserAccount(param.getUserId());

        List<BubbleTaskItemV3DTO> result = Lists.newArrayList();
        fillFreeBubble(result, freeBubble, userAccountDTO);
        fillVideoBubble(result, videoBubble, userAccountDTO);

        Collections.shuffle(result);
        return result;
    }

    private void fillVideoBubble(List<BubbleTaskItemV3DTO> result, int videoBubble, UserAccountDTO userAccountDTO) {
        if (videoBubble > 0) {
            Integer videoAwardNum = taskConfigFacadeService.getAwardConfig(
                userAccountDTO.getTotalFoods(),
                TaskActionEnum.BUBBLE_VIDEO_V3
            );
            // 填充看视频的气泡
            for (int i = 0; i < videoBubble; i++) {
                BubbleTaskItemV3DTO item = new BubbleTaskItemV3DTO();
                item.setTaskId(-1L);
                item.setAwardGrainNum(videoAwardNum);
                item.setAction(TaskActionEnum.BUBBLE_VIDEO_V3.name());
                item.setIcon(taskProperties.getVideoBubbleIconUrl());

                result.add(item);
            }
        }
    }

    private void fillFreeBubble(List<BubbleTaskItemV3DTO> result, int freeBubble, UserAccountDTO userAccountDTO) {
        if (freeBubble <= 0) {
            return;
        }

        Integer freeAwardNum = taskConfigFacadeService.getAwardConfig(
            userAccountDTO.getTotalFoods(),
            TaskActionEnum.BUBBLE_VIDEO_DIRECT_V3
        );
        // 填充免费领取的气泡
        for (int i = 0; i < freeBubble; i++) {
            BubbleTaskItemV3DTO item = new BubbleTaskItemV3DTO();
            item.setTaskId(-1L);
            item.setAwardGrainNum(freeAwardNum);
            item.setAction(TaskActionEnum.BUBBLE_VIDEO_DIRECT_V3.name());
            item.setIcon(taskProperties.getFreeBubbleIconUrl());

            result.add(item);
        }
    }

    @Override
    public List<TaskListItemV2DTO> fetchList(BaseUserParam param) {
        UserAccountDTO userAccountDTO = userAccountFacadeService.loadUserAccount(param.getUserId());
        Map<String, UserTaskItemBO> cacheTaskMap = taskCacheHolder.fetchUserTask(param.getUserId());

        UserTaskContext context = UserTaskContext.builder()
                                                 .param(param)
                                                 .userAccount(userAccountDTO)
                                                 .configService(taskConfigFacadeService)
                                                 .finalTaskList(Lists.newArrayList())
                                                 .taskCacheMap(cacheTaskMap)
                                                 .build();

        logicFilterChainExecutor.doFilter(LogicConstant.USER_TASK_FILTER, context);

        List<TaskListItemV2DTO> finalTaskList = context.getFinalTaskList();
        for (TaskListItemV2DTO taskListItemV2DTO : finalTaskList) {
            if (taskListItemV2DTO.getFinish()) {
                taskListItemV2DTO.setTaskClickAction("NONE");
                // 已完成的任务移动到后面
                if (taskListItemV2DTO.getOrder() <= 0) {
                    taskListItemV2DTO.setOrder(1);
                }
                taskListItemV2DTO.setOrder(taskListItemV2DTO.getOrder() * 100);
            }
        }
        return finalTaskList.stream().sorted().collect(Collectors.toList());
    }
}
